%%
%% NCCPS.cls 2018/08/26 
%% 
%% This is the NCCPS LaTeX class for authors of the National Conference on Chemical Process Simulation (NCCPS) .
%% 
%%
%%
%%
%% 
%%*************************************************************************
%% Legal Notice:
%% This code is offered as-is without any warranty either expressed or
%% implied; without even the implied warranty of MERCHANTABILITY or
%% FITNESS FOR A PARTICULAR PURPOSE! 
%% User assumes all risk.
%% In no event shall the NCCPS or any contributor to this code be liable for
%% any damages or losses, including, but not limited to, incidental,
%% consequential, or any other damages, resulting from the use or misuse
%% of any information contained here.
%%
%% All comments are the opinions of their respective authors and are not
%% necessarily endorsed by the NCCPS.
%%
%% This work is distributed under the LaTeX Project Public License (LPPL)
%% ( http://www.latex-project.org/ ) version 1.3, and may be freely used,
%% distributed and modified. A copy of the LPPL, version 1.3, is included
%% in the base LaTeX documentation of all distributions of LaTeX released
%% 2003/12/01 or later.
%% Retain all contribution notices and credits.
%% ** Modified files should be clearly indicated as such, including  **
%% ** renaming them and changing author support contact information. **
%%
%% File list of work: NCCPS.cls
%%                    bare_conf.tex, bare_jrnl.tex, bare_conf_compsoc.tex,
%%                    bare_jrnl_compsoc.tex
%% 
%%*************************************************************************
%%
%
% Available class options 
% e.g., \documentclass[10pt,conference]{NCCPS} 
% 
%             *** choose only one from each category ***
%
% 9pt, 10pt, 11pt, 12pt
%    Sets normal font size. The default is 10pt.
% 
% conference, journal, technote, peerreview, peerreviewca
%    determines format mode - conference papers, journal papers,
%    correspondence papers (technotes), or peer review papers. The user
%    should also select 9pt when using technote. peerreview is like
%    journal mode, but provides for a single-column "cover" title page for
%    anonymous peer review. The paper title (without the author names) is
%    repeated at the top of the page after the cover page. For peer review
%    papers, the \NCCPSpeerreviewmaketitle command must be executed (will
%    automatically be ignored for non-peerreview modes) at the place the
%    cover page is to end, usually just after the abstract (keywords are
%    not normally used with peer review papers). peerreviewca is like
%    peerreview, but allows the author names to be entered and formatted
%    as with conference mode so that author affiliation and contact
%    information can be easily seen on the cover page.
%    The default is journal.
%
% draft, draftcls, draftclsnofoot, final
%    determines if paper is formatted as a widely spaced draft (for
%    handwritten editor comments) or as a properly typeset final version.
%    draftcls restricts draft mode to the class file while all other LaTeX
%    packages (i.e., \usepackage{graphicx}) will behave as final - allows
%    for a draft paper with visible figures, etc. draftclsnofoot is like
%    draftcls, but does not display the date and the word "DRAFT" at the foot
%    of the pages. If using one of the draft modes, the user will probably
%    also want to select onecolumn.
%    The default is final.
%
% letterpaper, a4paper, cspaper
%    determines paper size: 8.5in X 11in, 210mm X 297mm or 7.875in X 10.75in.
%    Changing the paper size in the standard journal and conference modes
%    will not alter the typesetting of the document - only the margins will
%    be affected. In particular, documents using the a4paper option will
%    have reduced side margins (A4 is narrower than US letter) and a longer
%    bottom margin (A4 is longer than US letter). For both cases, the top
%    margins will be the same and the text will be horizontally centered.
%    For the compsoc conference and draft modes, it is the margins that will
%    remain constant, and thus the text area size will vary, with changes in
%    the paper size.
%    The cspaper option is the special ``trim'' paper size (7.875in x 10.75in)
%    used in the actual publication of Computer Society journals. Under
%    compsoc journal mode, this option does not alter the typesetting of the
%    document. Authors should invoke the cspaper option only if requested to
%    do so by the editors of the specific journal they are submitting to.
%    For final submission to the NCCPS, authors should generally use US letter 
%    (8.5 X 11in) paper unless otherwise instructed. Note that authors should
%    ensure that all post-processing (ps, pdf, etc.) uses the same paper
%    specificiation as the .tex document. Problems here are by far the number
%    one reason for incorrect margins. NCCPS will automatically set the
%    default paper size under pdflatex (without requiring any change to
%    pdftex.cfg), so this issue is more important to dvips users. Fix
%    config.ps, config.pdf, or ~/.dvipsrc for dvips, or use the
%    dvips -t papersize option instead as needed.
%
% oneside, twoside
%    determines if layout follows single sided or two sided (duplex)
%    printing. The only notable change is with the headings at the top of
%    the pages.
%    The default is oneside.
%
% onecolumn, twocolumn
%    determines if text is organized into one or two columns per page. One
%    column mode is usually used only with draft papers.
%    The default is twocolumn.
%
%
% romanappendices
%    Use the "Appendix I" convention when numbering appendices. NCCPS.cls
%    now defaults to Alpha "Appendix A" convention 
% captionsoff
%    disables the display of the figure/table captions. Some  journals
%    request that captions be removed and figures/tables be put on pages
%    of their own at the end of an initial paper submission. The endfloat
%    package can be used with this class option to achieve this format.
%
% nofonttune
%    turns off tuning of the font interword spacing. Maybe useful to those
%    not using the standard Times fonts or for those who have already "tuned"
%    their fonts.
%    The default is to enable NCCPS to tune font parameters.
%
%
%----------
% Available CLASSINPUTs provided (all are macros unless otherwise noted):
% \CLASSINPUTbaselinestretch
% \CLASSINPUTinnersidemargin
% \CLASSINPUToutersidemargin
% \CLASSINPUTtoptextmargin
% \CLASSINPUTbottomtextmargin
%
% Available CLASSINFOs provided:
% \ifCLASSINFOpdf                       (TeX if conditional)
% \CLASSINFOpaperwidth                  (macro)
% \CLASSINFOpaperheight                 (macro)
% \CLASSINFOnormalsizebaselineskip      (length)
% \CLASSINFOnormalsizeunitybaselineskip (length)
%
% Available CLASSOPTIONs provided:
% all class option flags (TeX if conditionals) unless otherwise noted,
% e.g., \ifCLASSOPTIONcaptionsoff
% point size options provided as a single macro:
% \CLASSOPTIONpt
% which will be defined as 9, 10, 11, or 12 depending on the document's
% normalsize point size.
% also, class option peerreviewca implies the use of class option peerreview
% and classoption draft implies the use of class option draftcls


\ProvidesClass{NCCPS}
\NeedsTeXFormat{LaTeX2e}


% These do nothing, but provide them like in article.cls
\newif\if@restonecol
\newif\if@titlepage


% class option conditionals
\newif\ifCLASSOPTIONonecolumn       \CLASSOPTIONonecolumnfalse
\newif\ifCLASSOPTIONtwocolumn       \CLASSOPTIONtwocolumntrue

\newif\ifCLASSOPTIONoneside         \CLASSOPTIONonesidetrue
\newif\ifCLASSOPTIONtwoside         \CLASSOPTIONtwosidefalse

\newif\ifCLASSOPTIONfinal           \CLASSOPTIONfinaltrue
\newif\ifCLASSOPTIONdraft           \CLASSOPTIONdraftfalse
\newif\ifCLASSOPTIONdraftcls        \CLASSOPTIONdraftclsfalse
\newif\ifCLASSOPTIONdraftclsnofoot  \CLASSOPTIONdraftclsnofootfalse

\newif\ifCLASSOPTIONpeerreview      \CLASSOPTIONpeerreviewfalse
\newif\ifCLASSOPTIONpeerreviewca    \CLASSOPTIONpeerreviewcafalse

\newif\ifCLASSOPTIONjournal         \CLASSOPTIONjournaltrue
\newif\ifCLASSOPTIONconference      \CLASSOPTIONconferencefalse
\newif\ifCLASSOPTIONtechnote        \CLASSOPTIONtechnotefalse

\newif\ifCLASSOPTIONnofonttune      \CLASSOPTIONnofonttunefalse

\newif\ifCLASSOPTIONcaptionsoff     \CLASSOPTIONcaptionsofffalse

\newif\ifCLASSOPTIONcomsoc          \CLASSOPTIONcomsocfalse
\newif\ifCLASSOPTIONcompsoc         \CLASSOPTIONcompsocfalse
\newif\ifCLASSOPTIONtransmag        \CLASSOPTIONtransmagfalse

\newif\ifCLASSOPTIONromanappendices \CLASSOPTIONromanappendicesfalse


% class info conditionals

% indicates if pdf (via pdflatex) output
\newif\ifCLASSINFOpdf               \CLASSINFOpdffalse


% V1.6b internal flag to show if using a4paper
\newif\if@NCCPSusingAfourpaper       \@NCCPSusingAfourpaperfalse
% V1.6b internal flag to show if using cspaper
\newif\if@NCCPSusingcspaper          \@NCCPSusingcspaperfalse


% NCCPS class scratch pad registers
% dimen
\newdimen\@NCCPStmpdimenA
\newdimen\@NCCPStmpdimenB
\newdimen\@NCCPStmpdimenC
% count
\newcount\@NCCPStmpcountA
\newcount\@NCCPStmpcountB
\newcount\@NCCPStmpcountC
% token list
\newtoks\@NCCPStmptoksA

% we use \CLASSOPTIONpt so that we can ID the point size (even for 9pt docs)
% as well as LaTeX's \@ptsize to retain some compatability with some
% external packages
\def\@ptsize{0}
% LaTeX does not support 9pt, so we set \@ptsize to 0 - same as that of 10pt
\DeclareOption{9pt}{\def\CLASSOPTIONpt{9}\def\@ptsize{0}}
\DeclareOption{10pt}{\def\CLASSOPTIONpt{10}\def\@ptsize{0}}
\DeclareOption{11pt}{\def\CLASSOPTIONpt{11}\def\@ptsize{1}}
\DeclareOption{12pt}{\def\CLASSOPTIONpt{12}\def\@ptsize{2}}



\DeclareOption{letterpaper}{\setlength{\paperwidth}{8.5in}%
                            \setlength{\paperheight}{11in}%
                            \@NCCPSusingAfourpaperfalse
                            \@NCCPSusingcspaperfalse
                            \def\CLASSOPTIONpaper{letter}%
                            \def\CLASSINFOpaperwidth{8.5in}%
                            \def\CLASSINFOpaperheight{11in}}


\DeclareOption{a4paper}{\setlength{\paperwidth}{210mm}%
                        \setlength{\paperheight}{297mm}%
                        \@NCCPSusingAfourpapertrue
                        \@NCCPSusingcspaperfalse
                        \def\CLASSOPTIONpaper{a4}%
                        \def\CLASSINFOpaperwidth{210mm}%
                        \def\CLASSINFOpaperheight{297mm}}

% special paper option for compsoc journals
\DeclareOption{cspaper}{\setlength{\paperwidth}{7.875in}%
                        \setlength{\paperheight}{10.75in}%
                        \@NCCPSusingcspapertrue
                        \@NCCPSusingAfourpaperfalse
                        \def\CLASSOPTIONpaper{NCCPScs}%
                        \def\CLASSINFOpaperwidth{7.875in}%
                        \def\CLASSINFOpaperheight{10.75in}}

\DeclareOption{oneside}{\@twosidefalse\@mparswitchfalse
                        \CLASSOPTIONonesidetrue\CLASSOPTIONtwosidefalse}
\DeclareOption{twoside}{\@twosidetrue\@mparswitchtrue
                        \CLASSOPTIONtwosidetrue\CLASSOPTIONonesidefalse}

\DeclareOption{onecolumn}{\CLASSOPTIONonecolumntrue\CLASSOPTIONtwocolumnfalse}
\DeclareOption{twocolumn}{\CLASSOPTIONtwocolumntrue\CLASSOPTIONonecolumnfalse}

% If the user selects draft, then this class AND any packages
% will go into draft mode.
\DeclareOption{draft}{\CLASSOPTIONdrafttrue\CLASSOPTIONdraftclstrue
                      \CLASSOPTIONdraftclsnofootfalse} 
% draftcls is for a draft mode which will not affect any packages
% used by the document.
\DeclareOption{draftcls}{\CLASSOPTIONdraftfalse\CLASSOPTIONdraftclstrue
                         \CLASSOPTIONdraftclsnofootfalse} 
% draftclsnofoot is like draftcls, but without the footer.
\DeclareOption{draftclsnofoot}{\CLASSOPTIONdraftfalse\CLASSOPTIONdraftclstrue
                               \CLASSOPTIONdraftclsnofoottrue} 
\DeclareOption{final}{\CLASSOPTIONdraftfalse\CLASSOPTIONdraftclsfalse
                      \CLASSOPTIONdraftclsnofootfalse}

\DeclareOption{journal}{\CLASSOPTIONpeerreviewfalse\CLASSOPTIONpeerreviewcafalse
                        \CLASSOPTIONjournaltrue\CLASSOPTIONconferencefalse\CLASSOPTIONtechnotefalse}

\DeclareOption{conference}{\CLASSOPTIONpeerreviewfalse\CLASSOPTIONpeerreviewcafalse
                           \CLASSOPTIONjournalfalse\CLASSOPTIONconferencetrue\CLASSOPTIONtechnotefalse}

\DeclareOption{technote}{\CLASSOPTIONpeerreviewfalse\CLASSOPTIONpeerreviewcafalse
                         \CLASSOPTIONjournalfalse\CLASSOPTIONconferencefalse\CLASSOPTIONtechnotetrue}

\DeclareOption{peerreview}{\CLASSOPTIONpeerreviewtrue\CLASSOPTIONpeerreviewcafalse
                           \CLASSOPTIONjournalfalse\CLASSOPTIONconferencefalse\CLASSOPTIONtechnotefalse}

\DeclareOption{peerreviewca}{\CLASSOPTIONpeerreviewtrue\CLASSOPTIONpeerreviewcatrue
                             \CLASSOPTIONjournalfalse\CLASSOPTIONconferencefalse\CLASSOPTIONtechnotefalse}

\DeclareOption{nofonttune}{\CLASSOPTIONnofonttunetrue}

\DeclareOption{captionsoff}{\CLASSOPTIONcaptionsofftrue}

\DeclareOption{comsoc}{\CLASSOPTIONcomsoctrue\CLASSOPTIONcompsocfalse\CLASSOPTIONtransmagfalse}

\DeclareOption{compsoc}{\CLASSOPTIONcomsocfalse\CLASSOPTIONcompsoctrue\CLASSOPTIONtransmagfalse}

\DeclareOption{transmag}{\CLASSOPTIONtransmagtrue\CLASSOPTIONcomsocfalse\CLASSOPTIONcompsocfalse}

\DeclareOption{romanappendices}{\CLASSOPTIONromanappendicestrue}


% default to US letter paper, 10pt, twocolumn, one sided, final, journal
\ExecuteOptions{letterpaper,10pt,twocolumn,oneside,final,journal}
% overrride these defaults per user requests
\ProcessOptions



%% -- Command Argument Scanning Support Functions --

% Sets the category codes for punctuation to their normal values.
% For local use with argument scanning.
\def\NCCPSnormalcatcodespunct{\catcode`\!=12 \catcode`\,=12 \catcode`\:=12
\catcode`\;=12 \catcode`\`=12 \catcode`\'=12 \catcode`\"=12 \catcode`\.=12
\catcode`\/=12 \catcode`\?=12 \catcode`\*=12 \catcode`\+=12 \catcode`\-=12
\catcode`\<=12 \catcode`\>=12 \catcode`\(=12 \catcode`\)=12 \catcode`\[=12
\catcode`\]=12 \catcode`\==12 \catcode`\|=12}
% Sets the category codes for numbers to their normal values.
% For local use with argument scanning.
\def\NCCPSnormalcatcodesnum{\catcode`\0=12 \catcode`\1=12 \catcode`\2=12
\catcode`\3=12 \catcode`\4=12 \catcode`\5=12 \catcode`\6=12 \catcode`\7=12
\catcode`\8=12 \catcode`\9=12}
% combined action of \NCCPSnormalcatcodespunct and \NCCPSnormalcatcodesnum
\def\NCCPSnormalcatcodes{\NCCPSnormalcatcodespunct\NCCPSnormalcatcodesnum}


% usage: \@NCCPSextracttoken*{}
% \@NCCPSextracttoken fully expands its argument (which it then stores in
% \@NCCPSextracttokenarg) via \edef and then the meaning of the first
% nonbrace (but including the empty group) token found is assigned via \let
% to \@NCCPSextractedtoken as well as stored in the macro
% \@NCCPSextractedtokenmacro. Tokens that would otherwise be discarded during
% the acquisition of the first are stored in \@NCCPSextractedtokensdiscarded,
% however their original relative brace nesting depths are not guaranteed to
% be preserved.
% If the argument is empty, or if a first nonbrace token does not exist (or
% is an empty group), \@NCCPSextractedtoken will be \relax and
% \@NCCPSextractedtokenmacro and \@NCCPSextractedtokensdiscarded will be empty.
%
% For example:
% \@NCCPSextracttoken{{{ab}{cd}}{{ef}g}}
% results in:
%
% \@NCCPSextracttokenarg          ==> a macro containing {{ab}{cd}}{{ef}g}
% \@NCCPSextractedtoken           ==> the letter a
% \@NCCPSextractedtokenmacro      ==> a macro containing a
% \@NCCPSextractedtokensdiscarded ==> a macro containing bcd{ef}g
%
% the *-star form, \@NCCPSextracttoken*, does not expand its argument
% contents during processing.
\def\@NCCPSextracttoken{\@ifstar{\let\@NCCPSextracttokendef=\def\@@NCCPSextracttoken}{\let\@NCCPSextracttokendef=\edef\@@NCCPSextracttoken}}

\def\@@NCCPSextracttoken#1{\@NCCPSextracttokendef\@NCCPSextracttokenarg{#1}\relax
\def\@NCCPSextractedtokensdiscarded{}\relax % initialize to empty
% if the macro is unchanged after being acquired as a single undelimited argument
% with anything after it being stripped off as a delimited argument
% we know we have one token without any enclosing braces. loop until this is true.
\let\@NCCPSextracttokencurgroup\@NCCPSextracttokenarg
\loop
  % trap case of an empty argument as this would cause a problem with
  % \@@@NCCPSextracttoken's first (nondelimited) argument acquisition
  \ifx\@NCCPSextracttokencurgroup\@empty
    \def\@NCCPSextractedtokenmacro{}\relax
  \else
    \expandafter\@@@NCCPSextracttoken\@NCCPSextracttokencurgroup\@NCCPSgeneralsequenceDELIMITER\relax
  \fi
  \ifx\@NCCPSextractedtokenmacro\@NCCPSextracttokencurgroup
  \else
    \let\@NCCPSextracttokencurgroup=\@NCCPSextractedtokenmacro
\repeat
% we can safely do a \let= here because there should be at most one token
% the relax is needed to handle the case of no token found
\expandafter\let\expandafter\@NCCPSextractedtoken\@NCCPSextractedtokenmacro\relax}

\def\@@@NCCPSextracttoken#1#2\@NCCPSgeneralsequenceDELIMITER{\def\@NCCPSextractedtokenmacro{#1}\relax
\def\@@NCCPSextractedtokensdiscarded{#2}\expandafter\expandafter\expandafter\def\expandafter\expandafter\expandafter
\@NCCPSextractedtokensdiscarded\expandafter\expandafter\expandafter
{\expandafter\@@NCCPSextractedtokensdiscarded\@NCCPSextractedtokensdiscarded}}
%%
%% -- End of Command Argument Scanning Support Functions --



% Computer Society conditional execution command
\long\def\@NCCPScompsoconly#1{\relax\ifCLASSOPTIONcompsoc\relax#1\relax\fi\relax}
% inverse
\long\def\@NCCPSnotcompsoconly#1{\relax\ifCLASSOPTIONcompsoc\else\relax#1\relax\fi\relax}
% compsoc conference
\long\def\@NCCPScompsocconfonly#1{\relax\ifCLASSOPTIONcompsoc\ifCLASSOPTIONconference\relax#1\relax\fi\fi\relax}
% compsoc not conference
\long\def\@NCCPScompsocnotconfonly#1{\relax\ifCLASSOPTIONcompsoc\ifCLASSOPTIONconference\else\relax#1\relax\fi\fi\relax}


% comsoc verify that newtxmath, mtpro2, mt11p or mathtime has been loaded
\def\@NCCPScomsocverifymathfont{\typeout{-- Verifying Times compatible math font.}\relax
  \@ifpackageloaded{newtxmath}{\typeout{-- newtxmath loaded, OK.}}{\@@NCCPScomsocverifymathfont}}
\def\@@NCCPScomsocverifymathfont{\@ifpackageloaded{mtpro2}{\typeout{-- mtpro2 loaded, OK.}}{\@@@NCCPScomsocverifymathfont}}
\def\@@@NCCPScomsocverifymathfont{\@ifpackageloaded{mt11p}{\typeout{-- mt11p2 loaded, OK.}}{\@@@@NCCPScomsocverifymathfont}}
\def\@@@@NCCPScomsocverifymathfont{\@ifpackageloaded{mathtime}{\typeout{-- mathtime loaded, OK.}}{\@NCCPScomsocenforcemathfont}}

% comsoc, if a Times math font was not loaded by user, enforce it
\def\@NCCPScomsocenforcemathfont{\typeout{** Times compatible math font not found, forcing.}\relax
\IfFileExists{newtxmath.sty}{\typeout{-- Found newtxmath, loading.}\RequirePackage{newtxmath}}{\@@NCCPScomsocenforcemathfont}}
\def\@@NCCPScomsocenforcemathfont{\IfFileExists{mtpro2.sty}{\typeout{-- Found mtpro2, loading.}\RequirePackage{mtpro2}}{\@@@NCCPScomsocenforcemathfont}}
\def\@@@NCCPScomsocenforcemathfont{\IfFileExists{mt11p.sty}{\typeout{-- Found mt11p, loading.}\RequirePackage{mt11p}}{\@@@@NCCPScomsocenforcemathfont}}
\def\@@@@NCCPScomsocenforcemathfont{\IfFileExists{mathtime.sty}{\typeout{-- Found mathtime, loading.}\RequirePackage{mathtime}}{\@@@@@NCCPScomsocenforcemathfont}}
% if no acceptable Times math font package found, error with newtxmath requirement
\def\@@@@@NCCPScomsocenforcemathfont{\typeout{** No Times compatible math font package found. newtxmath is required.}\RequirePackage{newtxmath}}


\ifCLASSOPTIONcomsoc
  % ensure that if newtxmath is used, the cmintegrals option is also invoked
  \PassOptionsToPackage{cmintegrals}{newtxmath}
  % comsoc requires a Times like math font
  % ensure this requirement is satisfied at document start
  \AtBeginDocument{\@NCCPScomsocverifymathfont}
\fi



% The NCCPS uses Times Roman font, so we'll default to Times.
% These three commands make up the entire times.sty package.
\renewcommand{\sfdefault}{phv}
\renewcommand{\rmdefault}{ptm}
\renewcommand{\ttdefault}{pcr}

% V1.7 compsoc nonconference papers, use Palatino/Palladio as the main text font,
% not Times Roman.
\@NCCPScompsocnotconfonly{\renewcommand{\rmdefault}{ppl}}

% enable the selected main text font
\normalfont\selectfont


\ifCLASSOPTIONcomsoc
  \typeout{-- Using NCCPS Communications Society mode.}
\fi

\ifCLASSOPTIONcompsoc
  \typeout{-- Using NCCPS Computer Society mode.}
\fi


% V1.7 conference notice message hook
\def\@NCCPSconsolenoticeconference{\typeout{}%
\typeout{** Conference Paper **}%
\typeout{Before submitting the final camera ready copy, remember to:}%
\typeout{}%
\typeout{ 1. Manually equalize the lengths of two columns on the last page}%
\typeout{ of your paper;}%
\typeout{}%
\typeout{ 2. Ensure that any PostScript and/or PDF output post-processing}%
\typeout{ uses only Type 1 fonts and that every step in the generation}%
\typeout{ process uses the appropriate paper size.}%
\typeout{}}


% we can send console reminder messages to the user here
\AtEndDocument{\ifCLASSOPTIONconference\@NCCPSconsolenoticeconference\fi}


% warn about the use of single column other than for draft mode
\ifCLASSOPTIONtwocolumn\else%
  \ifCLASSOPTIONdraftcls\else%
   \typeout{** ATTENTION: Single column mode is not typically used with NCCPS publications.}%
  \fi%
\fi


% V1.7 improved paper size setting code.
% Set pdfpage and dvips paper sizes. Conditional tests are similar to that
% of ifpdf.sty. Retain within {} to ensure tested macros are never altered,
% even if only effect is to set them to \relax.
% if \pdfoutput is undefined or equal to relax, output a dvips special
{\@ifundefined{pdfoutput}{\AtBeginDvi{\special{papersize=\CLASSINFOpaperwidth,\CLASSINFOpaperheight}}}{%
% pdfoutput is defined and not equal to \relax
% check for pdfpageheight existence just in case someone sets pdfoutput
% under non-pdflatex. If exists, set them regardless of value of \pdfoutput.
\@ifundefined{pdfpageheight}{\relax}{\global\pdfpagewidth\paperwidth
\global\pdfpageheight\paperheight}%
% if using \pdfoutput=0 under pdflatex, send dvips papersize special
\ifcase\pdfoutput
\AtBeginDvi{\special{papersize=\CLASSINFOpaperwidth,\CLASSINFOpaperheight}}%
\else
% we are using pdf output, set CLASSINFOpdf flag
\global\CLASSINFOpdftrue
\fi}}

% let the user know the selected papersize
\typeout{-- Using \CLASSINFOpaperwidth\space x \CLASSINFOpaperheight\space
(\CLASSOPTIONpaper)\space paper.}

\ifCLASSINFOpdf
\typeout{-- Using PDF output.}
\else
\typeout{-- Using DVI output.}
\fi


% The idea hinted here is for LaTeX to generate markleft{} and markright{}
% automatically for you after you enter \author{}, \journal{},
% \journaldate{}, journalvol{}, \journalnum{}, etc.
% However, there may be some backward compatibility issues here as
% well as some special applications for NCCPS.cls and special issues
% that may require the flexible \markleft{}, \markright{} and/or \markboth{}.
% We'll leave this as an open future suggestion.
%\newcommand{\journal}[1]{\def\@journal{#1}}
%\def\@journal{}



% pointsize values
% used with ifx to determine the document's normal size
\def\@NCCPSptsizenine{9}
\def\@NCCPSptsizeten{10}
\def\@NCCPSptsizeeleven{11}
\def\@NCCPSptsizetwelve{12}



% FONT DEFINITIONS (No sizexx.clo file needed) 
% V1.6 revised font sizes, displayskip values and
%      revised normalsize baselineskip to reduce underfull vbox problems
%      on the 58pc = 696pt = 9.5in text height we want
%      normalsize     #lines/column  baselineskip (aka leading)
%             9pt     63             11.0476pt (truncated down)
%            10pt     58             12pt      (exact)
%            11pt     52             13.3846pt (truncated down)
%            12pt     50             13.92pt   (exact)
%

% we need to store the nominal baselineskip for the given font size
% in case baselinestretch ever changes.
% this is a dimen, so it will not hold stretch or shrink
\newdimen\@NCCPSnormalsizeunitybaselineskip
\@NCCPSnormalsizeunitybaselineskip\baselineskip



%% ******* WARNING! *******
%%
%% Authors should not alter font sizes, baselineskip ("leading"),
%% margins or other spacing values in an attempt to squeeze more
%% material on each page.
%%
%% The NCCPS's own typesetting software will restore the correct
%% values when re-typesetting/proofing the submitted document,
%% possibly resulting in unexpected article over length charges.
%%
%% ******* WARNING! *******


% 9pt option defaults
\ifx\CLASSOPTIONpt\@NCCPSptsizenine
\typeout{-- This is a 9 point document.}
\def\normalsize{\@setfontsize{\normalsize}{9}{11.0476pt}}
\setlength{\@NCCPSnormalsizeunitybaselineskip}{11.0476pt}
\normalsize
\abovedisplayskip 1.5ex plus 3pt minus 1pt
\belowdisplayskip \abovedisplayskip
\abovedisplayshortskip 0pt plus 3pt
\belowdisplayshortskip 1.5ex plus 3pt minus 1pt
\def\small{\@setfontsize{\small}{8.5}{10pt}}
\def\footnotesize{\@setfontsize{\footnotesize}{8}{9pt}}
\def\scriptsize{\@setfontsize{\scriptsize}{7}{8pt}}
\def\tiny{\@setfontsize{\tiny}{5}{6pt}}
% sublargesize is the same as large - 10pt
\def\sublargesize{\@setfontsize{\sublargesize}{10}{12pt}}
\def\large{\@setfontsize{\large}{10}{12pt}}
\def\Large{\@setfontsize{\Large}{12}{14pt}}
\def\LARGE{\@setfontsize{\LARGE}{14}{17pt}}
\def\huge{\@setfontsize{\huge}{17}{20pt}}
\def\Huge{\@setfontsize{\Huge}{20}{24pt}}
\fi
%
% 10pt option defaults
\ifx\CLASSOPTIONpt\@NCCPSptsizeten
\typeout{-- This is a 10 point document.}
\def\normalsize{\@setfontsize{\normalsize}{10}{12.00pt}}
\setlength{\@NCCPSnormalsizeunitybaselineskip}{12pt}
\normalsize
\abovedisplayskip 1.5ex plus 4pt minus 2pt
\belowdisplayskip \abovedisplayskip
\abovedisplayshortskip 0pt plus 4pt
\belowdisplayshortskip 1.5ex plus 4pt minus 2pt
\def\small{\@setfontsize{\small}{9}{10pt}}
\def\footnotesize{\@setfontsize{\footnotesize}{8}{9pt}}
\def\scriptsize{\@setfontsize{\scriptsize}{7}{8pt}}
\def\tiny{\@setfontsize{\tiny}{5}{6pt}}
% sublargesize is a tad smaller than large - 11pt
\def\sublargesize{\@setfontsize{\sublargesize}{11}{13.4pt}}
\def\large{\@setfontsize{\large}{12}{14pt}}
\def\Large{\@setfontsize{\Large}{14}{17pt}}
\def\LARGE{\@setfontsize{\LARGE}{17}{20pt}}
\def\huge{\@setfontsize{\huge}{20}{24pt}}
\def\Huge{\@setfontsize{\Huge}{24}{28pt}}
\fi
%
% 11pt option defaults
\ifx\CLASSOPTIONpt\@NCCPSptsizeeleven
\typeout{-- This is an 11 point document.}
\def\normalsize{\@setfontsize{\normalsize}{11}{13.3846pt}}
\setlength{\@NCCPSnormalsizeunitybaselineskip}{13.3846pt}
\normalsize
\abovedisplayskip 1.5ex plus 5pt minus 3pt
\belowdisplayskip \abovedisplayskip
\abovedisplayshortskip 0pt plus 5pt
\belowdisplayshortskip 1.5ex plus 5pt minus 3pt
\def\small{\@setfontsize{\small}{10}{12pt}}
\def\footnotesize{\@setfontsize{\footnotesize}{9}{10.5pt}}
\def\scriptsize{\@setfontsize{\scriptsize}{8}{9pt}}
\def\tiny{\@setfontsize{\tiny}{6}{7pt}}
% sublargesize is the same as large - 12pt
\def\sublargesize{\@setfontsize{\sublargesize}{12}{14pt}}
\def\large{\@setfontsize{\large}{12}{14pt}}
\def\Large{\@setfontsize{\Large}{14}{17pt}}
\def\LARGE{\@setfontsize{\LARGE}{17}{20pt}}
\def\huge{\@setfontsize{\huge}{20}{24pt}}
\def\Huge{\@setfontsize{\Huge}{24}{28pt}}
\fi
%
% 12pt option defaults
\ifx\CLASSOPTIONpt\@NCCPSptsizetwelve
\typeout{-- This is a 12 point document.}
\def\normalsize{\@setfontsize{\normalsize}{12}{13.92pt}}
\setlength{\@NCCPSnormalsizeunitybaselineskip}{13.92pt}
\normalsize
\abovedisplayskip 1.5ex plus 6pt minus 4pt
\belowdisplayskip \abovedisplayskip
\abovedisplayshortskip 0pt plus 6pt
\belowdisplayshortskip 1.5ex plus 6pt minus 4pt
\def\small{\@setfontsize{\small}{10}{12pt}}
\def\footnotesize{\@setfontsize{\footnotesize}{9}{10.5pt}}
\def\scriptsize{\@setfontsize{\scriptsize}{8}{9pt}}
\def\tiny{\@setfontsize{\tiny}{6}{7pt}}
% sublargesize is the same as large - 14pt
\def\sublargesize{\@setfontsize{\sublargesize}{14}{17pt}}
\def\large{\@setfontsize{\large}{14}{17pt}}
\def\Large{\@setfontsize{\Large}{17}{20pt}}
\def\LARGE{\@setfontsize{\LARGE}{20}{24pt}}
\def\huge{\@setfontsize{\huge}{22}{26pt}}
\def\Huge{\@setfontsize{\Huge}{24}{28pt}}
\fi



% V1.8a compsoc font sizes
% compsoc font sizes use bp "Postscript" point units (1/72in) 
% rather than the traditional pt (1/72.27)
\ifCLASSOPTIONcompsoc
% -- compsoc defaults --
% ** will override some of these values later **
% 9pt
\ifx\CLASSOPTIONpt\@NCCPSptsizenine
\def\normalsize{\@setfontsize{\normalsize}{9bp}{11bp}}
\setlength{\@NCCPSnormalsizeunitybaselineskip}{11bp}
\normalsize
\abovedisplayskip 1.5ex plus 3bp minus 1bp
\belowdisplayskip \abovedisplayskip
\abovedisplayshortskip 0bp plus 3bp
\belowdisplayshortskip 1.5ex plus 3bp minus 1bp
\def\small{\@setfontsize{\small}{8.5bp}{10bp}}
\def\footnotesize{\@setfontsize{\footnotesize}{8bp}{9bp}}
\def\scriptsize{\@setfontsize{\scriptsize}{7bp}{8bp}}
\def\tiny{\@setfontsize{\tiny}{5bp}{6bp}}
% sublargesize is the same as large - 10bp
\def\sublargesize{\@setfontsize{\sublargesize}{10bp}{12bp}}
\def\large{\@setfontsize{\large}{10bp}{12bp}}
\def\Large{\@setfontsize{\Large}{12bp}{14bp}}
\def\LARGE{\@setfontsize{\LARGE}{14bp}{17bp}}
\def\huge{\@setfontsize{\huge}{17bp}{20bp}}
\def\Huge{\@setfontsize{\Huge}{20bp}{24bp}}
\fi
%
% 10pt
\ifx\CLASSOPTIONpt\@NCCPSptsizeten
\def\normalsize{\@setfontsize{\normalsize}{10bp}{12bp}}
\setlength{\@NCCPSnormalsizeunitybaselineskip}{12bp}
\normalsize
\abovedisplayskip 1.5ex plus 4bp minus 2bp
\belowdisplayskip \abovedisplayskip
\abovedisplayshortskip 0pt plus 4bp
\belowdisplayshortskip 1.5ex plus 4bp minus 2bp
\def\small{\@setfontsize{\small}{9bp}{10bp}}
\def\footnotesize{\@setfontsize{\footnotesize}{8bp}{9bp}}
\def\scriptsize{\@setfontsize{\scriptsize}{7bp}{8bp}}
\def\tiny{\@setfontsize{\tiny}{5bp}{6bp}}
% sublargesize is a tad smaller than large - 11bp
\def\sublargesize{\@setfontsize{\sublargesize}{11bp}{13.5bp}}
\def\large{\@setfontsize{\large}{12bp}{14bp}}
\def\Large{\@setfontsize{\Large}{14bp}{17bp}}
\def\LARGE{\@setfontsize{\LARGE}{17bp}{20bp}}
\def\huge{\@setfontsize{\huge}{20bp}{24bp}}
\def\Huge{\@setfontsize{\Huge}{24bp}{28bp}}
\fi
%
% 11pt
\ifx\CLASSOPTIONpt\@NCCPSptsizeeleven
\def\normalsize{\@setfontsize{\normalsize}{11bp}{13.5bp}}
\setlength{\@NCCPSnormalsizeunitybaselineskip}{13.5bp}
\normalsize
\abovedisplayskip 1.5ex plus 5bp minus 3bp
\belowdisplayskip \abovedisplayskip
\abovedisplayshortskip 0pt plus 5bp
\belowdisplayshortskip 1.5ex plus 5bp minus 3bp
\def\small{\@setfontsize{\small}{10bp}{12bp}}
\def\footnotesize{\@setfontsize{\footnotesize}{9bp}{10.5bp}}
\def\scriptsize{\@setfontsize{\scriptsize}{8bp}{9bp}}
\def\tiny{\@setfontsize{\tiny}{6bp}{7bp}}
% sublargesize is the same as large - 12bp
\def\sublargesize{\@setfontsize{\sublargesize}{12bp}{14bp}}
\def\large{\@setfontsize{\large}{12bp}{14bp}}
\def\Large{\@setfontsize{\Large}{14bp}{17bp}}
\def\LARGE{\@setfontsize{\LARGE}{17bp}{20bp}}
\def\huge{\@setfontsize{\huge}{20bp}{24bp}}
\def\Huge{\@setfontsize{\Huge}{24bp}{28bp}}
\fi
%
% 12pt
\ifx\CLASSOPTIONpt\@NCCPSptsizetwelve
\def\normalsize{\@setfontsize{\normalsize}{12bp}{14bp}}%
\setlength{\@NCCPSnormalsizeunitybaselineskip}{14bp}%
\normalsize
\abovedisplayskip 1.5ex plus 6bp minus 4bp
\belowdisplayskip \abovedisplayskip
\abovedisplayshortskip 0pt plus 6bp
\belowdisplayshortskip 1.5ex plus 6bp minus 4bp
\def\small{\@setfontsize{\small}{10bp}{12bp}}
\def\footnotesize{\@setfontsize{\footnotesize}{9bp}{10.5bp}}
\def\scriptsize{\@setfontsize{\scriptsize}{8bp}{9bp}}
\def\tiny{\@setfontsize{\tiny}{6bp}{7bp}}
% sublargesize is the same as large - 14bp
\def\sublargesize{\@setfontsize{\sublargesize}{14bp}{17bp}}
\def\large{\@setfontsize{\large}{14bp}{17bp}}
\def\Large{\@setfontsize{\Large}{17bp}{20bp}}
\def\LARGE{\@setfontsize{\LARGE}{20bp}{24bp}}
\def\huge{\@setfontsize{\huge}{22bp}{26bp}}
\def\Huge{\@setfontsize{\Huge}{24bp}{28bp}}
\fi
%
% -- override defaults: compsoc journals use special normalsizes --
\ifCLASSOPTIONconference
% 
% compsoc conferences
% 9pt
\ifx\CLASSOPTIONpt\@NCCPSptsizenine
\def\normalsize{\@setfontsize{\normalsize}{9bp}{10.8bp}}
\setlength{\@NCCPSnormalsizeunitybaselineskip}{10.8bp}
\normalsize
\abovedisplayskip 1.5ex plus 3bp minus 1bp
\belowdisplayskip \abovedisplayskip
\abovedisplayshortskip 0bp plus 3bp
\belowdisplayshortskip 1.5ex plus 3bp minus 1bp
\fi
% 10pt
\ifx\CLASSOPTIONpt\@NCCPSptsizeten
\def\normalsize{\@setfontsize{\normalsize}{10bp}{11.2bp}}
\setlength{\@NCCPSnormalsizeunitybaselineskip}{11.2bp}
\normalsize
\abovedisplayskip 1.5ex plus 4bp minus 2bp
\belowdisplayskip \abovedisplayskip
\abovedisplayshortskip 0pt plus 4bp
\belowdisplayshortskip 1.5ex plus 4bp minus 2bp
\fi
% 11pt
\ifx\CLASSOPTIONpt\@NCCPSptsizeeleven
\def\normalsize{\@setfontsize{\normalsize}{11bp}{13.2bp}}
\setlength{\@NCCPSnormalsizeunitybaselineskip}{13.2bp}
\normalsize
\abovedisplayskip 1.5ex plus 5bp minus 3bp
\belowdisplayskip \abovedisplayskip
\abovedisplayshortskip 0pt plus 5bp
\belowdisplayshortskip 1.5ex plus 5bp minus 3bp
\fi
% 12pt
\ifx\CLASSOPTIONpt\@NCCPSptsizetwelve
\def\normalsize{\@setfontsize{\normalsize}{12bp}{14.4bp}}
\setlength{\@NCCPSnormalsizeunitybaselineskip}{14.4bp}
\normalsize
\abovedisplayskip 1.5ex plus 6bp minus 4bp
\belowdisplayskip \abovedisplayskip
\abovedisplayshortskip 0pt plus 6bp
\belowdisplayshortskip 1.5ex plus 6bp minus 4bp
\fi
% 
% compsoc nonconferences
\else
% 9pt
\ifx\CLASSOPTIONpt\@NCCPSptsizenine
\def\normalsize{\@setfontsize{\normalsize}{9bp}{10.8bp}}
\setlength{\@NCCPSnormalsizeunitybaselineskip}{10.8bp}
\normalsize
\abovedisplayskip 1.5ex plus 3bp minus 1bp
\belowdisplayskip \abovedisplayskip
\abovedisplayshortskip 0bp plus 3bp
\belowdisplayshortskip 1.5ex plus 3bp minus 1bp
\fi
% 10pt
\ifx\CLASSOPTIONpt\@NCCPSptsizeten
% the official spec is 9.5bp with 11.4bp leading for 10pt,
% but measurements of proofs suggest upto 11.723bp leading
% here we'll use 11.54bp which gives 61 lines per column
% with the standard compsoc margins
\def\normalsize{\@setfontsize{\normalsize}{9.5bp}{11.54bp}}
\setlength{\@NCCPSnormalsizeunitybaselineskip}{11.54bp}
\normalsize
\abovedisplayskip 1.5ex plus 4bp minus 2bp
\belowdisplayskip \abovedisplayskip
\abovedisplayshortskip 0pt plus 4bp
\belowdisplayshortskip 1.5ex plus 4bp minus 2bp
\fi
% 11pt
\ifx\CLASSOPTIONpt\@NCCPSptsizeeleven
\def\normalsize{\@setfontsize{\normalsize}{11bp}{13.2bp}}
\setlength{\@NCCPSnormalsizeunitybaselineskip}{13.2bp}
\normalsize
\abovedisplayskip 1.5ex plus 5bp minus 3bp
\belowdisplayskip \abovedisplayskip
\abovedisplayshortskip 0pt plus 5bp
\belowdisplayshortskip 1.5ex plus 5bp minus 3bp
\fi
% 12pt
\ifx\CLASSOPTIONpt\@NCCPSptsizetwelve
\def\normalsize{\@setfontsize{\normalsize}{12bp}{14.4bp}}
\setlength{\@NCCPSnormalsizeunitybaselineskip}{14.4bp}
\normalsize
\abovedisplayskip 1.5ex plus 6bp minus 4bp
\belowdisplayskip \abovedisplayskip
\abovedisplayshortskip 0pt plus 6bp
\belowdisplayshortskip 1.5ex plus 6bp minus 4bp
\fi
\fi\fi




% V1.6 The Computer Modern Fonts will issue a substitution warning for
% 24pt titles (24.88pt is used instead, but the default and correct
% Times font will scale exactly as needed) increase the substitution
% tolerance to turn off this warning.
% 
% V1.8a, the compsoc bp font sizes can also cause bogus font substitution
% warnings with footnote or scriptsize math and the $\bullet$ itemized
% list of \NCCPScompsocitemizethanks. So, increase this to 1.5pt or more.
\def\fontsubfuzz{1.7bp}


% warn the user in case they forget to use the 9pt option with
% technote
\ifCLASSOPTIONtechnote%
 \ifx\CLASSOPTIONpt\@NCCPSptsizenine\else%
  \typeout{** ATTENTION: Technotes are normally 9pt documents.}%
 \fi%
\fi


% V1.7
% Improved \textunderscore to provide a much better fake _ when used with
% OT1 encoding. Under OT1, detect use of pcr or cmtt \ttfamily and use
% available true _ glyph for those two typewriter fonts.
\def\@NCCPSstringptm{ptm} % Times Roman family
\def\@NCCPSstringppl{ppl} % Palatino Roman family
\def\@NCCPSstringphv{phv} % Helvetica Sans Serif family
\def\@NCCPSstringpcr{pcr} % Courier typewriter family
\def\@NCCPSstringcmtt{cmtt} % Computer Modern typewriter family
\DeclareTextCommandDefault{\textunderscore}{\leavevmode
\ifx\f@family\@NCCPSstringpcr\string_\else
\ifx\f@family\@NCCPSstringcmtt\string_\else
\ifx\f@family\@NCCPSstringptm\kern 0em\vbox{\hrule\@width 0.5em\@height 0.5pt\kern -0.3ex}\else
\ifx\f@family\@NCCPSstringppl\kern 0em\vbox{\hrule\@width 0.5em\@height 0.5pt\kern -0.3ex}\else
\ifx\f@family\@NCCPSstringphv\kern -0.03em\vbox{\hrule\@width 0.62em\@height 0.52pt\kern -0.33ex}\kern -0.03em\else
\kern 0.09em\vbox{\hrule\@width 0.6em\@height 0.44pt\kern -0.63pt\kern -0.42ex}\kern 0.09em\fi\fi\fi\fi\fi\relax}




% set the default \baselinestretch
\def\baselinestretch{1}
\ifCLASSOPTIONdraftcls
  \def\baselinestretch{1.5}% default baselinestretch for draft modes
\fi 


% process CLASSINPUT baselinestretch
\ifx\CLASSINPUTbaselinestretch\@NCCPSundefined
\else
  \edef\baselinestretch{\CLASSINPUTbaselinestretch} % user CLASSINPUT override
  \typeout{** ATTENTION: Overriding \string\baselinestretch\space to
           \baselinestretch\space via \string\CLASSINPUT.}
\fi

\small\normalsize % make \baselinestretch take affect




% store the normalsize baselineskip
\newdimen\CLASSINFOnormalsizebaselineskip
\CLASSINFOnormalsizebaselineskip=\baselineskip\relax
% and the normalsize unity (baselinestretch=1) baselineskip
% we could save a register by giving the user access to
% \@NCCPSnormalsizeunitybaselineskip. However, let's protect
% its read only internal status
\newdimen\CLASSINFOnormalsizeunitybaselineskip
\CLASSINFOnormalsizeunitybaselineskip=\@NCCPSnormalsizeunitybaselineskip\relax
% store the nominal value of jot
\newdimen\NCCPSnormaljot
\NCCPSnormaljot=0.25\baselineskip\relax

% set \jot
\jot=\NCCPSnormaljot\relax




% V1.6, we are now going to fine tune the interword spacing
% The default interword glue for Times under TeX appears to use a
% nominal interword spacing of 25% (relative to the font size, i.e., 1em)
% a maximum of 40% and a minimum of 19%.
% For example, 10pt text uses an interword glue of:
% 
% 2.5pt plus 1.49998pt minus 0.59998pt
% 
% However, the NCCPS allows for a more generous range which reduces the need
% for hyphenation, especially for two column text. Furthermore, the NCCPS
% tends to use a little bit more nominal space between the words.
% The NCCPS's interword spacing percentages appear to be:
% 35% nominal
% 23% minimum
% 50% maximum
% (They may even be using a tad more for the largest fonts such as 24pt.)
% 
% for bold text, the NCCPS increases the spacing a little more:
% 37.5% nominal
% 23% minimum
% 55% maximum

% here are the interword spacing ratios we'll use
% for medium (normal weight)
\def\@NCCPSinterspaceratioM{0.35}
\def\@NCCPSinterspaceMINratioM{0.23}
\def\@NCCPSinterspaceMAXratioM{0.50}

% for bold
\def\@NCCPSinterspaceratioB{0.375}
\def\@NCCPSinterspaceMINratioB{0.23}
\def\@NCCPSinterspaceMAXratioB{0.55}


% compsoc nonconference papers use Palatino,
% tweak settings to better match the proofs
\ifCLASSOPTIONcompsoc
\ifCLASSOPTIONconference\else
% for medium (normal weight)
\def\@NCCPSinterspaceratioM{0.28}
\def\@NCCPSinterspaceMINratioM{0.21}
\def\@NCCPSinterspaceMAXratioM{0.47}
% for bold
\def\@NCCPSinterspaceratioB{0.305}
\def\@NCCPSinterspaceMINratioB{0.21}
\def\@NCCPSinterspaceMAXratioB{0.52}
\fi\fi


% command to revise the interword spacing for the current font under TeX:
% \fontdimen2 = nominal interword space
% \fontdimen3 = interword stretch
% \fontdimen4 = interword shrink
% since all changes to the \fontdimen are global, we can enclose these commands
% in braces to confine any font attribute or length changes
\def\@@@NCCPSsetfontdimens#1#2#3{{%
\setlength{\@NCCPStmpdimenB}{\f@size pt}% grab the font size in pt, could use 1em instead.
\setlength{\@NCCPStmpdimenA}{#1\@NCCPStmpdimenB}%
\fontdimen2\font=\@NCCPStmpdimenA\relax
\addtolength{\@NCCPStmpdimenA}{-#2\@NCCPStmpdimenB}%
\fontdimen3\font=-\@NCCPStmpdimenA\relax
\setlength{\@NCCPStmpdimenA}{#1\@NCCPStmpdimenB}%
\addtolength{\@NCCPStmpdimenA}{-#3\@NCCPStmpdimenB}%
\fontdimen4\font=\@NCCPStmpdimenA\relax}}

% revise the interword spacing for each font weight
\def\@@NCCPSsetfontdimens{{%
\mdseries
\@@@NCCPSsetfontdimens{\@NCCPSinterspaceratioM}{\@NCCPSinterspaceMAXratioM}{\@NCCPSinterspaceMINratioM}%
\bfseries
\@@@NCCPSsetfontdimens{\@NCCPSinterspaceratioB}{\@NCCPSinterspaceMAXratioB}{\@NCCPSinterspaceMINratioB}%
}}

% revise the interword spacing for each font shape
% \slshape is not often used for NCCPS work and is not altered here. The \scshape caps are
% already a tad too large in the free LaTeX fonts (as compared to what the NCCPS uses) so we
% won't alter these either.
\def\@NCCPSsetfontdimens{{%
\normalfont
\@@NCCPSsetfontdimens
\normalfont\itshape
\@@NCCPSsetfontdimens
}}

% command to revise the interword spacing for each font size (and shape
% and weight). Only the \rmfamily is done here as \ttfamily uses a 
% fixed spacing and \sffamily is not used as the main text of NCCPS papers.
\def\@NCCPStunefonts{{\selectfont\rmfamily
\tiny\@NCCPSsetfontdimens
\scriptsize\@NCCPSsetfontdimens
\footnotesize\@NCCPSsetfontdimens
\small\@NCCPSsetfontdimens
\normalsize\@NCCPSsetfontdimens
\sublargesize\@NCCPSsetfontdimens
\large\@NCCPSsetfontdimens
\LARGE\@NCCPSsetfontdimens
\huge\@NCCPSsetfontdimens
\Huge\@NCCPSsetfontdimens}}

% if the nofonttune class option is not given, revise the interword spacing
% now - in case NCCPS makes any default length measurements, and make
% sure all the default fonts are loaded
\ifCLASSOPTIONnofonttune\else
\@NCCPStunefonts
\fi

% and again at the start of the document in case the user loaded different fonts
\AtBeginDocument{\ifCLASSOPTIONnofonttune\else\@NCCPStunefonts\fi}





% -- V1.8a page setup commands --

% The default sample text for calculating margins
% Note that NCCPS publications use \scriptsize for headers and footers.
\def\NCCPSdefaultsampletext{\normalfont\normalsize gT}
\def\NCCPSdefaultheadersampletext{\normalfont\scriptsize T}% NCCPS headers default to uppercase
\def\NCCPSdefaultfootersampletext{\normalfont\scriptsize gT}



% usage: \NCCPSsettextwidth{inner margin}{outer margin}
% Sets \textwidth to allow the specified inner and outer margins
% for the current \paperwidth.
\def\NCCPSsettextwidth#1#2{\@NCCPStmpdimenA\paperwidth
\@NCCPStmpdimenB#1\relax
\advance\@NCCPStmpdimenA by -\@NCCPStmpdimenB
\@NCCPStmpdimenB#2\relax
\advance\@NCCPStmpdimenA by -\@NCCPStmpdimenB
\textwidth\@NCCPStmpdimenA}



% usage: \NCCPSsetsidemargin{mode: i, o, c, a}{margin/offset}
% Sets \oddsidemargin and \evensidemargin to yield the specified margin
% of the given mode.
% The available modes are:
% i = inner margin
% o = outer margin
% c = centered, with the given offset
% a = adjust the margins using the given offset
% For the offsets, positive values increase the inner margin.
% \textwidth should be set properly for the given margins before calling this
% function.
\def\NCCPSsetsidemargin#1#2{\@NCCPStmpdimenA #2\relax
\@NCCPSextracttoken{#1}\relax
% check for mode errors
\ifx\@NCCPSextractedtokenmacro\@empty
  \@NCCPSclspkgerror{Empty mode type in \string\NCCPSsetsidemargin\space (line \the\inputlineno).\MessageBreak
  Defaulting to `i'}{Valid modes for \string\NCCPSsetsidemargin\space are: i, o, c and a.}\relax
  \let\@NCCPSextractedtoken=i\relax
  \def\@NCCPSextractedtokenmacro{i}\relax
\else
  \ifx\@NCCPSextractedtokensdiscarded\@empty\else
    \typeout{** WARNING: \string\NCCPSsetsidemargin\space mode specifiers after the first in `\@NCCPSextracttokenarg' ignored (line \the\inputlineno).}\relax
  \fi
\fi
% handle each mode
\if\@NCCPSextractedtoken a\relax
  \advance\oddsidemargin by \@NCCPStmpdimenA\relax
\else
\if\@NCCPSextractedtoken c\relax
  \oddsidemargin\paperwidth
  \advance\oddsidemargin by -\textwidth
  \divide\oddsidemargin by 2\relax
  \advance\oddsidemargin by -1in\relax
  \advance\oddsidemargin by \@NCCPStmpdimenA\relax
\else
\if\@NCCPSextractedtoken o\relax
  \oddsidemargin\paperwidth
  \advance\oddsidemargin by -\textwidth
  \advance\oddsidemargin by -\@NCCPStmpdimenA
  \advance\oddsidemargin by -1in\relax
\else
  \if\@NCCPSextractedtoken i\relax
  \else
    \@NCCPSclspkgerror{Unknown mode type `\@NCCPSextractedtokenmacro' in \string\NCCPSsetsidemargin\space (line \the\inputlineno).\MessageBreak
    Defaulting to `i'}%
    {Valid modes for \string\NCCPSsetsidemargin\space are: i, o, c  and a.}%
  \fi
  \oddsidemargin\@NCCPStmpdimenA
  \advance\oddsidemargin by -1in\relax
\fi\fi\fi
% odd and even side margins both mean "inner" for single sided pages
\evensidemargin\oddsidemargin
% but are mirrors of each other when twosided is in effect
\if@twoside
  \evensidemargin\paperwidth
  \advance\evensidemargin by -\textwidth
  \advance\evensidemargin by -\oddsidemargin
  % have to compensate for both the builtin 1in LaTex offset
  % and the fact we already subtracted this offset from \oddsidemargin
  \advance\evensidemargin -2in\relax
\fi}



% usage: \NCCPSsettextheight[sample text]{top text margin}{bottom text margin}
% Sets \textheight based on the specified top margin and bottom margin.
% Takes into consideration \paperheight, \topskip, and (by default) the
% the actual height and depth of the \NCCPSdefaultsampletext text. 
\def\NCCPSsettextheight{\@ifnextchar [{\@NCCPSsettextheight}{\@NCCPSsettextheight[\NCCPSdefaultsampletext]}}
\def\@NCCPSsettextheight[#1]#2#3{\textheight\paperheight\relax
  \@NCCPStmpdimenA #2\relax
  \advance \textheight by -\@NCCPStmpdimenA% subtract top margin
  \@NCCPStmpdimenA #3\relax
  \advance \textheight by -\@NCCPStmpdimenA% subtract bottom margin
  \advance \textheight by \topskip% add \topskip
  % subtract off everything above the top, and below the bottom, baselines
  \settoheight{\@NCCPStmpdimenA}{\begingroup #1\relax\relax\relax\endgroup}\relax
  \advance \textheight by -\@NCCPS
tmpdimenA
  \settodepth{\@NCCPStmpdimenA}{\begingroup #1\relax\relax\relax\endgroup}\relax
  \advance \textheight by -\@NCCPStmpdimenA}



\newdimen\NCCPSquantizedlength
\NCCPSquantizedlength 0sp\relax
\newdimen\NCCPSquantizedlengthdiff
\NCCPSquantizedlengthdiff 0sp\relax
\def\NCCPSquantizedlengthint{0}

% usage: \NCCPSquantizelength{mode: d, c, i}{base unit}{length}
% Sets the length \NCCPSquantizedlength to be an integer multiple of the given
% (nonzero) base unit such that \NCCPSquantizedlength approximates the given
% length.
% \NCCPSquantizedlengthdiff is a length equal to the difference between the
% \NCCPSquantizedlength and the given length.
% \NCCPSquantizedlengthint is a macro containing the integer number of base units
% in \NCCPSquantizedlength.
% i.e., \NCCPSquantizedlength =  \NCCPSquantizedlengthint * base unit
% The mode determines how \NCCPSquantizedlength is quantized:
% d = always decrease (always round down \NCCPSquantizeint)
% c = use the closest match
% i = always increase (always round up \NCCPSquantizeint)
% In anycase, if the given length is already quantized,
% \NCCPSquantizedlengthdiff will be set to zero.
\def\NCCPSquantizelength#1#2#3{\begingroup
% work in isolation so as not to externally disturb the \@NCCPStmp
% variables
% load the argument values indirectly via \NCCPSquantizedlengthdiff
% in case the user refers to our \@NCCPStmpdimenX, \NCCPSquantizedlength,
% etc. in the arguments. we also will work with these as counters,
% i.e., in sp units
% A has the base unit
\NCCPSquantizedlengthdiff #2\relax\relax\relax\relax
\@NCCPStmpcountA\NCCPSquantizedlengthdiff
% B has the input length
\NCCPSquantizedlengthdiff #3\relax\relax\relax\relax
\@NCCPStmpcountB\NCCPSquantizedlengthdiff
\@NCCPStmpdimenA\the\@NCCPStmpcountA sp\relax
\@NCCPStmpdimenB\the\@NCCPStmpcountB sp\relax
% \@NCCPStmpcountC will have the quantized int
% \NCCPSquantizedlength will have the quantized length
% \@NCCPStmpdimenC will have the quantized diff
% initialize them to zero as this is what will be
% exported if an error occurs
\@NCCPStmpcountC 0\relax
\NCCPSquantizedlength 0sp\relax
\@NCCPStmpdimenC 0sp\relax
% extract mode
\@NCCPSextracttoken{#1}\relax
% check for mode errors
\ifx\@NCCPSextractedtokenmacro\@empty
  \@NCCPSclspkgerror{Empty mode type in \string\NCCPSquantizelength\space (line \the\inputlineno).\MessageBreak
  Defaulting to `d'}{Valid modes for \string\NCCPSquantizelength\space are: d, c and i.}\relax
  \let\@NCCPSextractedtoken=d\relax
  \def\@NCCPSextractedtokenmacro{d}\relax
\else
  \ifx\@NCCPSextractedtokensdiscarded\@empty\else
    \typeout{** WARNING: \string\NCCPSquantizelength\space mode specifiers after the first in `\@NCCPSextracttokenarg' ignored (line \the\inputlineno).}\relax
  \fi
\fi
% check for base unit is zero error
\ifnum\@NCCPStmpcountA=0\relax
\@NCCPSclspkgerror{Base unit is zero in \string\NCCPSquantizelength\space (line \the\inputlineno).\MessageBreak
  \string\NCCPSquantizedlength\space and \string\NCCPSquantizedlengthdiff\space are set to zero}{Division by zero is not allowed.}\relax
\else% base unit is nonzero
  % \@NCCPStmpcountC carries the number of integer units
  % in the quantized length (integer length \ base)
  \@NCCPStmpcountC\@NCCPStmpcountB\relax
  \divide\@NCCPStmpcountC by \@NCCPStmpcountA\relax
  % \NCCPSquantizedlength has the (rounded down) quantized length
  % = base * int
  \NCCPSquantizedlength\@NCCPStmpdimenA\relax
  \multiply\NCCPSquantizedlength by \@NCCPStmpcountC\relax
  % \@NCCPStmpdimenC has the difference
  % = quantized length - length
  \@NCCPStmpdimenC\NCCPSquantizedlength\relax
  \advance\@NCCPStmpdimenC by -\@NCCPStmpdimenB\relax
  % trap special case of length being already quantized
  % to avoid a roundup under i option
  \ifdim\@NCCPStmpdimenC=0sp\relax
  \else % length not is already quantized
    % set dimenA to carry the upper quantized (absolute value) difference:
    % quantizedlength + base - length
    \advance\@NCCPStmpdimenA by \NCCPSquantizedlength\relax
    \advance\@NCCPStmpdimenA by -\@NCCPStmpdimenB\relax
    % set dimenB to carry the lower quantized (absolute value) difference:
    % length - quantizedlength
    \advance\@NCCPStmpdimenB by -\NCCPSquantizedlength\relax
    % handle each mode
    \if\@NCCPSextractedtoken c\relax
      % compare upper and lower amounts, select upper if lower > upper
      \ifdim\@NCCPStmpdimenB>\@NCCPStmpdimenA\relax
        % use upper
        \advance\NCCPSquantizedlength by \the\@NCCPStmpcountA sp\relax
        \advance\@NCCPStmpcountC by 1\relax
        \@NCCPStmpdimenC\@NCCPStmpdimenA
      \else% <=. uselower
        % no need to do anything for lower, use output values already setup
      \fi
    \else% not mode c
    \if\@NCCPSextractedtoken i\relax
      % always round up under i mode
      \advance\NCCPSquantizedlength by \the\@NCCPStmpcountA sp\relax
      \advance\@NCCPStmpcountC by 1\relax
      \@NCCPStmpdimenC\@NCCPStmpdimenA
    \else
      \if\@NCCPSextractedtoken d\relax
      \else
        \@NCCPSclspkgerror{Unknown mode type `\@NCCPSextractedtokenmacro' in \string\NCCPSquantizelength\space (line \the\inputlineno).\MessageBreak
        Defaulting to `d'}%
        {Valid modes for \string\NCCPSquantizelength\space are: d, c, and i.}\relax
      \fi % if d
      % no need to do anything for d, use output values already setup
    \fi\fi % if i, c
  \fi % if length is already quantized
\fi% if base unit is zero
% globally assign the results to macros we use here to escape the enclosing
% group without needing to call \global on any of the \@NCCPStmp variables.
% \@NCCPStmpcountC has the quantized int
% \NCCPSquantizedlength has the quantized length
% \@NCCPStmpdimenC has the quantized diff
\xdef\@NCCPSquantizedlengthintmacro{\the\@NCCPStmpcountC}\relax
\@NCCPStmpcountC\NCCPSquantizedlength\relax
\xdef\@NCCPSquantizedlengthmacro{\the\@NCCPStmpcountC}\relax
\@NCCPStmpcountC\@NCCPStmpdimenC\relax
\xdef\@NCCPSquantizedlengthdiffmacro{\the\@NCCPStmpcountC}\relax
\endgroup
% locally assign the outputs here from the macros
\expandafter\NCCPSquantizedlength\@NCCPSquantizedlengthmacro sp\relax
\expandafter\NCCPSquantizedlengthdiff\@NCCPSquantizedlengthdiffmacro sp\relax
\edef\NCCPSquantizedlengthint{\@NCCPSquantizedlengthintmacro}\relax}



\newdimen\NCCPSquantizedtextheightdiff
\NCCPSquantizedtextheightdiff 0sp\relax

% usage: \NCCPSquantizetextheight[base unit]{mode: d, c, i}
% Sets \textheight to be an integer multiple of the current \baselineskip
% (or the optionally specified base unit) plus the first (\topskip) line.
% \NCCPSquantizedtextheightdiff is a length equal to the difference between
% the new quantized and original \textheight.
% \NCCPSquantizedtextheightlpc is a macro containing the integer number of
% lines per column under the quantized \textheight. i.e.,
% \textheight = \NCCPSquantizedtextheightlpc * \baselineskip + \topskip
% The mode determines how \textheight is quantized:
% d = always decrease (always round down the number of lines per column)
% c = use the closest match
% i = always increase (always round up the number of lines per column)
% In anycase, if \textheight is already quantized, it will remain unchanged,
% and \NCCPSquantizedtextheightdiff will be set to zero.
% Depends on: \NCCPSquantizelength
\def\NCCPSquantizetextheight{\@ifnextchar [{\@NCCPSquantizetextheight}{\@NCCPSquantizetextheight[\baselineskip]}}
\def\@NCCPSquantizetextheight[#1]#2{\begingroup
% use our \NCCPSquantizedtextheightdiff as a scratch pad
% we need to subtract off \topskip before quantization
\NCCPSquantizedtextheightdiff\textheight
\advance\NCCPSquantizedtextheightdiff by -\topskip\relax
\NCCPSquantizelength{#2}{#1}{\NCCPSquantizedtextheightdiff}
% add back \topskip line
\advance\NCCPSquantizedlength by \topskip
\@NCCPStmpcountC\NCCPSquantizedlengthint\relax
\advance\@NCCPStmpcountC by 1\relax
% globally assign the results to macros we use here to escape the enclosing
% group without needing to call \global on any of the \@NCCPStmp variables.
\xdef\@NCCPSquantizedtextheightlpcmacro{\the\@NCCPStmpcountC}\relax
\@NCCPStmpcountC\NCCPSquantizedlength\relax
\xdef\@NCCPSquantizedtextheightmacro{\the\@NCCPStmpcountC}\relax
\@NCCPStmpcountC\NCCPSquantizedlengthdiff\relax
\xdef\@NCCPSquantizedtextheightdiffmacro{\the\@NCCPStmpcountC}\relax
\endgroup
% locally assign the outputs here from the macros
\textheight\@NCCPSquantizedtextheightmacro sp\relax
\NCCPSquantizedtextheightdiff\@NCCPSquantizedtextheightdiffmacro sp\relax
\edef\NCCPSquantizedtextheightlpc{\@NCCPSquantizedtextheightlpcmacro}}



% usage: \NCCPSsettopmargin[sample text]{mode: t, b, c, a, q}{margin/offset}
% Sets \topmargin based on the specified vertical margin.
% Takes into consideration the base 1in offset, \headheight, \headsep,
% \topskip, and (by default) the the actual height (or, for the bottom, depth)
% of the \NCCPSdefaultsampletext text.
% The available modes are:
% t = top margin
% b = bottom margin
% c = vertically centered, with the given offset
% a = adjust the vertical margins using the given offset
% q = adjust the margins using \NCCPSquantizedtextheightdiff and the given offset
% For the offsets, positive values increase the top margin.
% \headheight, \headsep, \topskip and \textheight should be set properly for the
% given margins before calling this function.
\def\NCCPSsettopmargin{\@ifnextchar [{\@NCCPSsettopmargin}{\@NCCPSsettopmargin[\NCCPSdefaultsampletext]}}
\def\@NCCPSsettopmargin[#1]#2#3{\@NCCPStmpdimenA #3\relax
\@NCCPSextracttoken{#2}\relax
% check for mode errors
\ifx\@NCCPSextractedtokenmacro\@empty
  \@NCCPSclspkgerror{Empty mode type in \string\NCCPSsettopmargin\space (line \the\inputlineno).\MessageBreak
  Defaulting to `t'}{Valid modes for \string\NCCPSsettopmargin\space are: t, b, c, a and q.}\relax
  \let\@NCCPSextractedtoken=t\relax
  \def\@NCCPSextractedtokenmacro{t}\relax
\else
  \ifx\@NCCPSextractedtokensdiscarded\@empty\else
    \typeout{** WARNING: \string\NCCPSsettopmargin\space mode specifiers after the first in `\@NCCPSextracttokenarg' ignored (line \the\inputlineno).}\relax
  \fi
\fi
% handle each mode
\if\@NCCPSextractedtoken a\relax
  \advance\topmargin by \@NCCPStmpdimenA\relax
\else
\if\@NCCPSextractedtoken q\relax
  % we need to adjust by half the \NCCPSquantizedtextheightdiff value
  \@NCCPStmpdimenB\NCCPSquantizedtextheightdiff\relax
  \divide\@NCCPStmpdimenB by 2\relax
  % a positive \NCCPSquantizedtextheightdiff means we need to reduce \topmargin
  % because \textheight has been lenghtened
  \advance\topmargin by -\@NCCPStmpdimenB\relax
  \advance\topmargin by \@NCCPStmpdimenA\relax
\else
\if\@NCCPSextractedtoken c\relax
  \topmargin\paperheight
  \advance\topmargin by -\textheight
  % \textheight includes \topskip, but we should not count topskip whitespace here, backout
  \advance \topmargin by \topskip
  \settoheight{\@NCCPStmpdimenB}{\begingroup #1\relax\relax\relax\endgroup}\relax
  \advance\topmargin by -\@NCCPStmpdimenB\relax
  \settodepth{\@NCCPStmpdimenB}{\begingroup #1\relax\relax\relax\endgroup}\relax
  \advance\topmargin by -\@NCCPStmpdimenB\relax
  \divide\topmargin by 2\relax
  \advance\topmargin by \@NCCPStmpdimenA\relax
\else
\if\@NCCPSextractedtoken b\relax
  \topmargin\paperheight
  \advance\topmargin by -\textheight
  % \textheight includes \topskip, but we should not count topskip whitespace here, backout
  \advance \topmargin by \topskip
  \settodepth{\@NCCPStmpdimenB}{\begingroup #1\relax\relax\relax\endgroup}\relax
  \advance\topmargin by -\@NCCPStmpdimenB\relax
  \advance\topmargin by -\@NCCPStmpdimenA\relax
\else
  \if\@NCCPSextractedtoken t\relax
  \else
    \@NCCPSclspkgerror{Unknown mode type `\@NCCPSextractedtokenmacro' in \string\NCCPSsettopmargin\space (line \the\inputlineno).\MessageBreak
    Defaulting to `t'}%
    {Valid modes for \string\NCCPSsettopmargin\space are: t, b, c, a and q.}\relax
  \fi
  \topmargin\@NCCPStmpdimenA\relax
  \settoheight{\@NCCPStmpdimenB}{\begingroup #1\relax\relax\relax\endgroup}\relax
  \advance\topmargin by \@NCCPStmpdimenB\relax
\fi\fi % if t, b, c
% convert desired top margin into actual \topmargin
% this is not done for the q or a modes because they are only adjustments
\advance \topmargin by -\topskip
\advance \topmargin by -1in
\advance \topmargin by -\headheight
\advance \topmargin by -\headsep
\fi\fi % if q, a
}



% usage: \NCCPSsetheadermargin[header sample][text sample]{mode: t, b, c, a}{margin/offset}
% Differentially adjusts \topmargin and \headsep (such that their sum is unchanged)
% based on the specified header margin.
% Takes into consideration the base 1in offset, \headheight, \topskip, and (by default)
% the actual height (or depth) of the \NCCPSdefaultheadersampletext and
% \NCCPSdefaultsampletext text.
% The available modes are:
% t = top margin (top of the header text to the top of the page)
% b = bottom margin (bottom of the header text to the top of the main text)
% c = vertically centered between the main text and the top of the page,
%     with the given offset
% a = adjust the vertical position using the given offset
% For the offsets, positive values move the header downward.
% \headheight, \headsep, \topskip and \topmargin should be set properly before
% calling this function.
\def\NCCPSsetheadermargin{\@ifnextchar [{\@NCCPSsetheadermargin}{\@NCCPSsetheadermargin[\NCCPSdefaultheadersampletext]}}
\def\@NCCPSsetheadermargin[#1]{\@ifnextchar [{\@@NCCPSsetheadermargin[#1]}{\@@NCCPSsetheadermargin[#1][\NCCPSdefaultsampletext]}}
\def\@@NCCPSsetheadermargin[#1][#2]#3#4{\@NCCPStmpdimenA #4\relax
\@NCCPSextracttoken{#3}\relax
% check for mode errors
\ifx\@NCCPSextractedtokenmacro\@empty
  \@NCCPSclspkgerror{Empty mode type in \string\NCCPSsetheadermargin\space (line \the\inputlineno).\MessageBreak
  Defaulting to `t'}{Valid modes for \string\NCCPSsetheadermargin\space are: t, b, c, and a.}\relax
  \let\@NCCPSextractedtoken=t\relax
  \def\@NCCPSextractedtokenmacro{t}\relax
\else
  \ifx\@NCCPSextractedtokensdiscarded\@empty\else
    \typeout{** WARNING: \string\NCCPSsetheadermargin\space mode specifiers after the first in `\@NCCPSextracttokenarg' ignored (line \the\inputlineno).}\relax
  \fi
\fi
% handle each mode
\if\@NCCPSextractedtoken a\relax
  % No need to do anything here and can pass through the adjustment
  % value as is. The end adjustment of \topmargin and \headsep will
  % do all that is needed
\else
\if\@NCCPSextractedtoken c\relax
  % get the bottom margin
  \@NCCPStmpdimenB\headsep\relax
  \settodepth{\@NCCPStmpdimenC}{\begingroup #1\relax\relax\relax\endgroup}\relax
  \advance\@NCCPStmpdimenB by -\@NCCPStmpdimenC
  \advance\@NCCPStmpdimenB by \topskip
  \settoheight{\@NCCPStmpdimenC}{\begingroup #2\relax\relax\relax\endgroup}\relax
  \advance\@NCCPStmpdimenB by -\@NCCPStmpdimenC
  % at this point \@NCCPStmpdimenB has the actual header bottom margin
  % subtract from it the top header margin
  \advance\@NCCPStmpdimenB -1in\relax % take into consideration the system 1in offset of the top margin
  \advance\@NCCPStmpdimenB by -\topmargin
  \advance\@NCCPStmpdimenB by -\headheight
  \settoheight{\@NCCPStmpdimenC}{\begingroup #1\relax\relax\relax\endgroup}\relax
  \advance\@NCCPStmpdimenB by \@NCCPStmpdimenC
  % at this point \@NCCPStmpdimenB has the difference between the bottom and top margins
  % we need to adjust by half this amount to center the header
  \divide\@NCCPStmpdimenB by 2\relax
  % and add to offset
  \advance\@NCCPStmpdimenA by \@NCCPStmpdimenB
\else
\if\@NCCPSextractedtoken b\relax
  \@NCCPStmpdimenB\headsep\relax
  \settodepth{\@NCCPStmpdimenC}{\begingroup #1\relax\relax\relax\endgroup}\relax
  \advance\@NCCPStmpdimenB by -\@NCCPStmpdimenC
  \advance\@NCCPStmpdimenB by \topskip
  \settoheight{\@NCCPStmpdimenC}{\begingroup #2\relax\relax\relax\endgroup}\relax
  \advance\@NCCPStmpdimenB by -\@NCCPStmpdimenC
  % at this point \@NCCPStmpdimenB has the actual header bottom margin
  % get the difference between the actual and the desired
  \advance\@NCCPStmpdimenB by -\@NCCPStmpdimenA
  \@NCCPStmpdimenA\@NCCPStmpdimenB
\else
  \if\@NCCPSextractedtoken t\relax
  \else
    \@NCCPSclspkgerror{Unknown mode type `\@NCCPSextractedtokenmacro' in \string\NCCPSsetheadermargin\space (line \the\inputlineno).\MessageBreak
    Defaulting to `t'}%
    {Valid modes for \string\NCCPSsetheadermargin\space are: t, b, c and a.}\relax
  \fi
  \@NCCPStmpdimenB 1in\relax % take into consideration the system 1in offset of the top margin
  \advance\@NCCPStmpdimenB by \topmargin
  \advance\@NCCPStmpdimenB by \headheight
  \settoheight{\@NCCPStmpdimenC}{\begingroup #1\relax\relax\relax\endgroup}\relax
  \advance\@NCCPStmpdimenB by -\@NCCPStmpdimenC
  % at this point \@NCCPStmpdimenB has the actual header top margin
  % get the difference between the desired and the actual
  \advance\@NCCPStmpdimenA by -\@NCCPStmpdimenB
\fi\fi % if t, b, c
\fi % if a
% advance \topmargin by the needed amount and reduce \headsep by the same
% so as not to disturb the location of the main text
\advance\topmargin by \@NCCPStmpdimenA\relax
\advance\headsep by -\@NCCPStmpdimenA\relax
}



% usage: \NCCPSsetfootermargin[footer sample][text sample]{mode: t, b, c, a}{margin/offset}
% Adjusts \footskip based on the specified footer margin.
% Takes into consideration the base 1in offset, \paperheight, \headheight,
% \headsep, \textheight and (by default) the actual height (or depth) of the
% \NCCPSdefaultfootersampletext and \NCCPSdefaultsampletext text.
% The available modes are:
% t = top margin (top of the footer text to the bottom of the main text)
% b = bottom margin (bottom of the footer text to the bottom of page)
% c = vertically centered between the main text and the bottom of the page,
%     with the given offset
% a = adjust the vertical position using the given offset
% For the offsets, positive values move the footer downward.
% \headheight, \headsep, \topskip, \topmargin, and \textheight should be set
% properly before calling this function.
\def\NCCPSsetfootermargin{\@ifnextchar [{\@NCCPSsetfootermargin}{\@NCCPSsetfootermargin[\NCCPSdefaultfootersampletext]}}
\def\@NCCPSsetfootermargin[#1]{\@ifnextchar [{\@@NCCPSsetfootermargin[#1]}{\@@NCCPSsetfootermargin[#1][\NCCPSdefaultsampletext]}}
\def\@@NCCPSsetfootermargin[#1][#2]#3#4{\@NCCPStmpdimenA #4\relax
\@NCCPSextracttoken{#3}\relax
% check for mode errors
\ifx\@NCCPSextractedtokenmacro\@empty
  \@NCCPSclspkgerror{Empty mode type in \string\NCCPSsetfootermargin\space (line \the\inputlineno).\MessageBreak
  Defaulting to `t'}{Valid modes for \string\NCCPSsetfootermargin\space are: t, b, c, and a.}\relax
  \let\@NCCPSextractedtoken=t\relax
  \def\@NCCPSextractedtokenmacro{t}\relax
\else
  \ifx\@NCCPSextractedtokensdiscarded\@empty\else
    \typeout{** WARNING: \string\NCCPSsetfootermargin\space mode specifiers after the first in `\@NCCPSextracttokenarg' ignored (line \the\inputlineno).}\relax
  \fi
\fi
% handle each mode
\if\@NCCPSextractedtoken a\relax
  % No need to do anything here and can pass through the adjustment
  % value as is. The end adjustment of \footskip will do all that
  % is needed
\else
\if\@NCCPSextractedtoken c\relax
  % calculate the bottom margin
  \@NCCPStmpdimenB 1in\relax % system 1in offset
  \advance\@NCCPStmpdimenB\topmargin\relax
  \advance\@NCCPStmpdimenB\headheight\relax
  \advance\@NCCPStmpdimenB\headsep\relax
  \advance\@NCCPStmpdimenB\textheight\relax
  \advance\@NCCPStmpdimenB\footskip\relax
  \settodepth{\@NCCPStmpdimenC}{\begingroup #1\relax\relax\relax\endgroup}\relax
  \advance\@NCCPStmpdimenC by \@NCCPStmpdimenB
  \@NCCPStmpdimenB\paperheight
  \advance\@NCCPStmpdimenB by -\@NCCPStmpdimenC
  % at this point \@NCCPStmpdimenB has the actual footer bottom margin
  % now subtract off the footer top margin
  \advance\@NCCPStmpdimenB -\footskip\relax
  \settodepth{\@NCCPStmpdimenC}{\begingroup #2\relax\relax\relax\endgroup}\relax
  \advance\@NCCPStmpdimenB by \@NCCPStmpdimenC
  \settoheight{\@NCCPStmpdimenC}{\begingroup #1\relax\relax\relax\endgroup}\relax
  \advance\@NCCPStmpdimenB by \@NCCPStmpdimenC
  % at this point \@NCCPStmpdimenB has the difference between the bottom
  % and top footer margins
  % our adjustment must be half this value to center the footer
  \divide\@NCCPStmpdimenB by 2\relax
  % add to the offset
  \advance\@NCCPStmpdimenA by \@NCCPStmpdimenB
\else
\if\@NCCPSextractedtoken b\relax
  % calculate the bottom margin
  \@NCCPStmpdimenB 1in\relax % system 1in offset
  \advance\@NCCPStmpdimenB\topmargin\relax
  \advance\@NCCPStmpdimenB\headheight\relax
  \advance\@NCCPStmpdimenB\headsep\relax
  \advance\@NCCPStmpdimenB\textheight\relax
  \advance\@NCCPStmpdimenB\footskip\relax
  \settodepth{\@NCCPStmpdimenC}{\begingroup #1\relax\relax\relax\endgroup}\relax
  \advance\@NCCPStmpdimenC by \@NCCPStmpdimenB
  \@NCCPStmpdimenB\paperheight
  \advance\@NCCPStmpdimenB by -\@NCCPStmpdimenC
  % at this point \@NCCPStmpdimenB has the actual footer bottom margin
  % get the difference between the actual and the desired
  \advance\@NCCPStmpdimenB by -\@NCCPStmpdimenA
  \@NCCPStmpdimenA\@NCCPStmpdimenB
\else
  \if\@NCCPSextractedtoken t\relax
  \else
    \@NCCPSclspkgerror{Unknown mode type `\@NCCPSextractedtokenmacro' in \string\NCCPSsetfootermargin\space (line \the\inputlineno).\MessageBreak
    Defaulting to `t'}%
    {Valid modes for \string\NCCPSsetfootermargin\space are: t, b, c and a.}\relax
  \fi
  \@NCCPStmpdimenB\footskip\relax
  \settodepth{\@NCCPStmpdimenC}{\begingroup #2\relax\relax\relax\endgroup}\relax
  \advance\@NCCPStmpdimenB by -\@NCCPStmpdimenC
  \settoheight{\@NCCPStmpdimenC}{\begingroup #1\relax\relax\relax\endgroup}\relax
  \advance\@NCCPStmpdimenB by -\@NCCPStmpdimenC
  % at this point \@NCCPStmpdimenB has the actual footer top margin
  % get the difference between the desired and the actual
  \advance\@NCCPStmpdimenA by -\@NCCPStmpdimenB
\fi\fi % if t, b, c
\fi % if a
% advance \footskip by the needed amount
\advance\footskip by \@NCCPStmpdimenA\relax
}

% -- End V1.8a page setup commands --





% V1.6
% LaTeX is a little to quick to use hyphenations
% So, we increase the penalty for their use and raise
% the badness level that triggers an underfull hbox
% warning. The author may still have to tweak things,
% but the appearance will be much better "right out
% of the box" than that under V1.5 and prior.
% TeX default is 50
\hyphenpenalty=750
\ifCLASSOPTIONcompsoc
\hyphenpenalty 500
\fi
% If we didn't adjust the interword spacing, 2200 might be better.
% The TeX default is 1000
\hbadness=1350
% The NCCPS does not use extra spacing after punctuation
\frenchspacing

% V1.7 increase this a tad to discourage equation breaks
\binoppenalty=1000 % default 700
\relpenalty=800     % default 500

% v1.8a increase these to discourage widows and orphans
\clubpenalty=1000          % default 150
\widowpenalty=1000         % default 150
\displaywidowpenalty=1000 % default  50


% margin note stuff
\marginparsep      10pt
\marginparwidth    20pt
\marginparpush     25pt


% if things get too close, go ahead and let them touch
\lineskip            0pt
\normallineskip      0pt
\lineskiplimit       0pt
\normallineskiplimit 0pt

% The distance from the lower edge of the text body to the
% footline
\footskip 0.4in

% normally zero, should be relative to font height.
% put in a little rubber to help stop some bad breaks (underfull vboxes)
\parskip 0ex plus 0.2ex minus 0.1ex

\parindent    1.0em
\ifCLASSOPTIONcompsoc
  \parindent    1.5em
\fi

\headheight   12pt
\headsep      18pt
% use the normal font baselineskip
% so that \topskip is unaffected by changes in \baselinestretch
\topskip=\@NCCPSnormalsizeunitybaselineskip


% V1.8 \maxdepth defaults to 4pt, but should be font size dependent
\maxdepth=0.5\@NCCPSnormalsizeunitybaselineskip
\textheight       58pc  % 9.63in, 696pt

% set the default top margin to 58pt
% which results in a \topmargin of -49.59pt for 10pt documents
\NCCPSsettopmargin{t}{58pt}
% tweak textheight to a perfect integer number of lines/column.
% standard is: 9pt/63 lpc; 10pt/58 lpc; 11pt/52 lpc; 12pt/50 lpc
\NCCPSquantizetextheight{c}
% tweak top margin so that the error is shared equally at the top and bottom
\NCCPSsettopmargin{q}{0sp}


\columnsep         1pc
\textwidth        43pc   % 2 x 21pc + 1pc = 43pc

% set the default side margins to center the text
\NCCPSsetsidemargin{c}{0pt}


% adjust margins for default conference mode
\ifCLASSOPTIONconference
 \textheight        9.25in % The standard for conferences (668.4975pt)
 \NCCPSsettopmargin{t}{0.75in}
 % tweak textheight to a perfect integer number of lines/page.
 % standard is: 9pt/61 lpc; 10pt/56 lpc; 11pt/50 lpc; 12pt/48 lpc
 \NCCPSquantizetextheight{c}
 % tweak top margin so that the error is shared equally at the top and bottom
 \NCCPSsettopmargin{q}{0sp}
\fi


% compsoc text sizes, margins and spacings
\ifCLASSOPTIONcompsoc
 \columnsep         12bp
 % CS specs for \textwdith are 6.875in
 % \textwidth         6.875in
 % however, measurements from proofs show they are using 3.5in columns
 \textwidth         7in
 \advance\textwidth by \columnsep
 % set the side margins to center the text
 \NCCPSsetsidemargin{c}{0pt}
 % top/bottom margins to center
 % could just set \textheight to 9.75in for all the different paper sizes
 % and then quantize, but we'll do it the long way here to allow for easy
 % future per-paper size adjustments
 \NCCPSsettextheight{0.625in}{0.625in}% 11in - 2 * 0.625in = 9.75in is the standard text height for compsoc journals
 \NCCPSsettopmargin{t}{0.625in}
 \if@NCCPSusingcspaper
   \NCCPSsettextheight{0.5in}{0.5in}% 10.75in - 2 * 0.5in = 9.75in
   \NCCPSsettopmargin{t}{0.5in}
 \fi
 \if@NCCPSusingAfourpaper
   \NCCPSsettextheight{24.675mm}{24.675mm}% 297mm - 2 * 24.675mm = 247.650mm (9.75in)
   \NCCPSsettopmargin{t}{24.675mm}
 \fi
 % tweak textheight to a perfect integer number of lines/page.
 % standard is: 9pt/65 lpc; 10pt/61 lpc; 11pt/53 lpc; 12pt/49 lpc
 \NCCPSquantizetextheight{c}
 % tweak top margin so that the error is shared equally at the top and bottom
 \NCCPSsettopmargin{q}{0sp}

% compsoc conference
 \ifCLASSOPTIONconference
  % compsoc conference use a larger value for columnsep
  \columnsep 0.25in
  \NCCPSsettextwidth{0.75in}{0.75in}
  % set the side margins to center the text (0.75in for letterpaper)
  \NCCPSsetsidemargin{c}{0pt}
  % compsoc conferences want 1in top and bottom margin
  \NCCPSsettextheight{1in}{1in}
  \NCCPSsettopmargin{t}{1in}
  % tweak textheight to a perfect integer number of lines/page.
  % standard is: 9pt/58 lpc; 10pt/53 lpc; 11pt/48 lpc; 12pt/46 lpc
  \NCCPSquantizetextheight{c}
  % tweak top margin so that the error is shared equally at the top and bottom
  \NCCPSsettopmargin{q}{0sp}
 \fi
\fi



% draft mode settings override that of all other modes
% provides a nice 1in margin all around the paper and extra
% space between the lines for editor's comments
\ifCLASSOPTIONdraftcls
  % we want 1in side margins regardless of paper type
  \NCCPSsettextwidth{1in}{1in}
  \NCCPSsetsidemargin{c}{0pt}
  % want 1in top and bottom margins
  \NCCPSsettextheight{1in}{1in}
  \NCCPSsettopmargin{t}{1in}
  % digitize textheight to be an integer number of lines.
  % this may cause the top and bottom margins to be off a tad
  \NCCPSquantizetextheight{c}
  % tweak top margin so that the error is shared equally at the top and bottom
  \NCCPSsettopmargin{q}{0sp}
\fi



% process CLASSINPUT inner/outer margin
% if inner margin defined, but outer margin not, set outer to inner.
\ifx\CLASSINPUTinnersidemargin\@NCCPSundefined
\else
  \ifx\CLASSINPUToutersidemargin\@NCCPSundefined
    \edef\CLASSINPUToutersidemargin{\CLASSINPUTinnersidemargin}
  \fi
\fi

\ifx\CLASSINPUToutersidemargin\@NCCPSundefined
\else
  % if outer margin defined, but inner margin not, set inner to outer.
  \ifx\CLASSINPUTinnersidemargin\@NCCPSundefined
    \edef\CLASSINPUTinnersidemargin{\CLASSINPUToutersidemargin}
  \fi
  \NCCPSsettextwidth{\CLASSINPUTinnersidemargin}{\CLASSINPUToutersidemargin}
  \NCCPSsetsidemargin{i}{\CLASSINPUTinnersidemargin}
  \typeout{** ATTENTION: Overriding inner side margin to \CLASSINPUTinnersidemargin\space and 
           outer side margin to \CLASSINPUToutersidemargin\space via \string\CLASSINPUT.}
\fi



% process CLASSINPUT top/bottom text margin
% if toptext margin defined, but bottomtext margin not, set bottomtext to toptext margin
\ifx\CLASSINPUTtoptextmargin\@NCCPSundefined
\else
  \ifx\CLASSINPUTbottomtextmargin\@NCCPSundefined
    \edef\CLASSINPUTbottomtextmargin{\CLASSINPUTtoptextmargin}
  \fi
\fi

\ifx\CLASSINPUTbottomtextmargin\@NCCPSundefined
\else
  % if bottomtext margin defined, but toptext margin not, set toptext to bottomtext margin
  \ifx\CLASSINPUTtoptextmargin\@NCCPSundefined
    \edef\CLASSINPUTtoptextmargin{\CLASSINPUTbottomtextmargin}
  \fi
  \NCCPSsettextheight{\CLASSINPUTtoptextmargin}{\CLASSINPUTbottomtextmargin}
  \NCCPSsettopmargin{t}{\CLASSINPUTtoptextmargin}
  \typeout{** ATTENTION: Overriding top text margin to \CLASSINPUTtoptextmargin\space and 
           bottom text margin to \CLASSINPUTbottomtextmargin\space via \string\CLASSINPUT.}
\fi



% default to center header and footer text in the margins
\NCCPSsetheadermargin{c}{0pt}
\NCCPSsetfootermargin{c}{0pt}

% adjust header and footer positions for compsoc journals
\ifCLASSOPTIONcompsoc
  \ifCLASSOPTIONjournal
    \NCCPSsetheadermargin{b}{\@NCCPSnormalsizeunitybaselineskip}
    \NCCPSsetfootermargin{t}{\@NCCPSnormalsizeunitybaselineskip}
  \fi
\fi


% V1.8a display lines per column info message on user's console
\def\NCCPSdisplayinfolinespercolumn{\@NCCPStmpdimenA=\textheight
% topskip represents only one line even if > baselineskip
\advance\@NCCPStmpdimenA by -1\topskip
\@NCCPStmpcountA=\@NCCPStmpdimenA
\@NCCPStmpcountB=\@NCCPStmpdimenA
\divide\@NCCPStmpcountB by \baselineskip
% need to add one line to include topskip (first) line
\advance\@NCCPStmpcountB by 1
% save lines per column value as text
\edef\@NCCPSnumlinespercolumninfotxt{\the\@NCCPStmpcountB}
% backout topskip advance to allow direct \@NCCPStmpcountA comparison
\advance\@NCCPStmpcountB by -1
% restore value as text height (without topskip) rather than just as number of lines
\multiply\@NCCPStmpcountB by \baselineskip
% is the column height an integer number of lines per column?
\ifnum\@NCCPStmpcountA=\@NCCPStmpcountB
\edef\@NCCPSnumlinespercolumnexactinfotxt{exact}
\else
\@NCCPStmpdimenA\@NCCPStmpcountA sp\relax
\advance\@NCCPStmpdimenA by -\@NCCPStmpcountB sp\relax
\edef\@NCCPSnumlinespercolumnexactinfotxt{approximate, difference = \the\@NCCPStmpdimenA}
\fi
\typeout{-- Lines per column: \@NCCPSnumlinespercolumninfotxt\space (\@NCCPSnumlinespercolumnexactinfotxt).}}
% delay execution till start of document to allow for user changes
\AtBeginDocument{\NCCPSdisplayinfolinespercolumn}



% LIST SPACING CONTROLS

% Controls the amount of EXTRA spacing
% above and below \trivlist 
% Both \list and IED lists override this.
% However, \trivlist will use this as will most
% things built from \trivlist like the \center
% environment.
\topsep           0.5\baselineskip

% Controls the additional spacing around lists preceded
% or followed by blank lines. the NCCPS does not increase
% spacing before or after paragraphs so it is set to zero.
% \z@ is the same as zero, but faster.
\partopsep          \z@

% Controls the spacing between paragraphs in lists. 
% The NCCPS does not increase spacing before or after paragraphs
% so this is also zero. 
% With NCCPS.cls, global changes to
% this value DO affect lists (but not IED lists).
\parsep             \z@

% Controls the extra spacing between list items. 
% The NCCPS does not put extra spacing between items.
% With NCCPS.cls, global changes to this value DO affect
% lists (but not IED lists).
\itemsep            \z@

% \itemindent is the amount to indent the FIRST line of a list
% item. It is auto set to zero within the \list environment. To alter
% it, you have to do so when you call the \list.
% However, the NCCPS uses this for the theorem environment
% There is an alternative value for this near \leftmargini below
\itemindent         -1em

% \leftmargin, the spacing from the left margin of the main text to
% the left of the main body of a list item is set by \list.
% Hence this statement does nothing for lists.
% But, quote and verse do use it for indention.
\leftmargin         2em

% we retain this stuff from the older NCCPS.cls so that \list
% will work the same way as before. However, itemize, enumerate and
% description (IED) could care less about what these are as they
% all are overridden.
\leftmargini        2em
%\itemindent         2em  % Alternative values: sometimes used.
%\leftmargini        0em
\leftmarginii       1em
\leftmarginiii    1.5em
\leftmarginiv     1.5em
\leftmarginv      1.0em
\leftmarginvi     1.0em
\labelsep         0.5em 
\labelwidth         \z@


% The old NCCPS.cls behavior of \list is retained.
% However, the new V1.3 IED list environments override all the
% @list stuff (\@listX is called within \list for the
% appropriate level just before the user's list_decl is called). 
% \topsep is now 2pt as the NCCPS puts a little extra space around
% lists - used by those non-IED macros that depend on \list.
% Note that \parsep and \itemsep are not redefined as in 
% the sizexx.clo \@listX (which article.cls uses) so global changes
% of these values DO affect \list
% 
\def\@listi{\leftmargin\leftmargini \topsep 2pt plus 1pt minus 1pt}
\let\@listI\@listi
\def\@listii{\leftmargin\leftmarginii\labelwidth\leftmarginii%
    \advance\labelwidth-\labelsep \topsep 2pt}
\def\@listiii{\leftmargin\leftmarginiii\labelwidth\leftmarginiii%
    \advance\labelwidth-\labelsep \topsep 2pt}
\def\@listiv{\leftmargin\leftmarginiv\labelwidth\leftmarginiv%
    \advance\labelwidth-\labelsep \topsep 2pt}
\def\@listv{\leftmargin\leftmarginv\labelwidth\leftmarginv%
    \advance\labelwidth-\labelsep \topsep 2pt}
\def\@listvi{\leftmargin\leftmarginvi\labelwidth\leftmarginvi%
    \advance\labelwidth-\labelsep \topsep 2pt}


% The NCCPS uses 5) not 5.
\def\labelenumi{\theenumi)}     \def\theenumi{\arabic{enumi}}

% The NCCPS uses a) not (a)
\def\labelenumii{\theenumii)}  \def\theenumii{\alph{enumii}}

% The NCCPS uses iii) not iii.
\def\labelenumiii{\theenumiii)} \def\theenumiii{\roman{enumiii}}

% The NCCPS uses A) not A.
\def\labelenumiv{\theenumiv)}   \def\theenumiv{\Alph{enumiv}}

% exactly the same as in article.cls
\def\p@enumii{\theenumi}
\def\p@enumiii{\theenumi(\theenumii)}
\def\p@enumiv{\p@enumiii\theenumiii}

% itemized list label styles
\def\labelitemi{$\scriptstyle\bullet$}
\def\labelitemii{\textbf{--}}
\def\labelitemiii{$\ast$}
\def\labelitemiv{$\cdot$}



% **** V1.3 ENHANCEMENTS ****
% Itemize, Enumerate and Description (IED) List Controls
% ***************************
% 
% 
% The NCCPS seems to use at least two different values by
% which ITEMIZED list labels are indented to the right
% For The Journal of Lightwave Technology (JLT) and The Journal
% on Selected Areas in Communications (JSAC), they tend to use
% an indention equal to \parindent. For Transactions on Communications
% they tend to indent ITEMIZED lists a little more--- 1.3\parindent.
% We'll provide both values here for you so that you can choose 
% which one you like in your document using a command such as:
% setlength{\NCCPSilabelindent}{\NCCPSilabelindentB}
\newdimen\NCCPSilabelindentA
\NCCPSilabelindentA \parindent

\newdimen\NCCPSilabelindentB
\NCCPSilabelindentB 1.3\parindent
% However, we'll default to using \parindent
% which makes more sense to me
\newdimen\NCCPSilabelindent
\NCCPSilabelindent \NCCPSilabelindentA


% This controls the default amount the enumerated list labels
% are indented to the right.
% Normally, this is the same as the paragraph indention
\newdimen\NCCPSelabelindent
\NCCPSelabelindent \parindent

% This controls the default amount the description list labels
% are indented to the right.
% Normally, this is the same as the paragraph indention
\newdimen\NCCPSdlabelindent
\NCCPSdlabelindent \parindent

% This is the value actually used within the IED lists.
% The IED environments automatically set its value to
% one of the three values above, so global changes do 
% not have any effect
\newdimen\NCCPSlabelindent
\NCCPSlabelindent \parindent

% The actual amount labels will be indented is
% \NCCPSlabelindent multiplied by the factor below
% corresponding to the level of nesting depth
% This provides a means by which the user can
% alter the effective \NCCPSlabelindent for deeper
% levels
% There may not be such a thing as correct "standard NCCPS"
% values. What the NCCPS actually does may depend on the specific
% circumstances.
% The first list level almost always has full indention.
% The second levels I've seen have only 75% of the normal indentation
% Three level or greater nestings are very rare. I am guessing
% that they don't use any indentation.
\def\NCCPSlabelindentfactori{1.0}   % almost always one
\def\NCCPSlabelindentfactorii{0.75} % 0.0 or 1.0 may be used in some cases
\def\NCCPSlabelindentfactoriii{0.0} % 0.75? 0.5? 0.0?
\def\NCCPSlabelindentfactoriv{0.0}
\def\NCCPSlabelindentfactorv{0.0}
\def\NCCPSlabelindentfactorvi{0.0}

% value actually used within IED lists, it is auto
% set to one of the 6 values above
% global changes here have no effect
\def\NCCPSlabelindentfactor{1.0}

% This controls the default spacing between the end of the IED
% list labels and the list text, when normal text is used for
% the labels.
% compsoc uses a larger value here, but we'll set that later
% in the class so that this code block area can be extracted
% as-is for NCCPStools.sty
\newdimen\NCCPSiednormlabelsep
\NCCPSiednormlabelsep 0.6em

% This controls the default spacing between the end of the IED
% list labels and the list text, when math symbols are used for
% the labels (nomenclature lists). The NCCPS usually increases the 
% spacing in these cases
\newdimen\NCCPSiedmathlabelsep
\NCCPSiedmathlabelsep 1.2em

% This controls the extra vertical separation put above and
% below each IED list. the NCCPS usually puts a little extra spacing
% around each list. However, this spacing is barely noticeable.
% compsoc uses a larger value here, but we'll set that later
% in the class so that this code block area can be extracted
% as-is for NCCPStools.sty
\newskip\NCCPSiedtopsep
\NCCPSiedtopsep 2pt plus 1pt minus 1pt


% This command is executed within each IED list environment
% at the beginning of the list. You can use this to set the 
% parameters for some/all your IED list(s) without disturbing 
% global parameters that affect things other than lists.
% i.e., renewcommand{\NCCPSiedlistdecl}{\setlength{\labelsep}{5em}}
% will alter the \labelsep for the next list(s) until 
% \NCCPSiedlistdecl is redefined. 
\def\NCCPSiedlistdecl{\relax}

% This command provides an easy way to set \leftmargin based
% on the \labelwidth, \labelsep and the argument \NCCPSlabelindent
% Usage: \NCCPScalcleftmargin{width-to-indent-the-label}
% output is in the \leftmargin variable, i.e., effectively:
% \leftmargin = argument + \labelwidth + \labelsep
% Note controlled spacing here, shield end of lines with %
\def\NCCPScalcleftmargin#1{\setlength{\leftmargin}{#1}%
\addtolength{\leftmargin}{\labelwidth}%
\addtolength{\leftmargin}{\labelsep}}

% This command provides an easy way to set \labelwidth to the
% width of the given text. It is the same as
% \settowidth{\labelwidth}{label-text}
% and useful as a shorter alternative.
% Typically used to set \labelwidth to be the width
% of the longest label in the list
\def\NCCPSsetlabelwidth#1{\settowidth{\labelwidth}{#1}}

% When this command is executed, IED lists will use the 
% NCCPSiedmathlabelsep label separation rather than the normal
% spacing. To have an effect, this command must be executed via
% the \NCCPSiedlistdecl or within the option of the IED list
% environments.
\def\NCCPSusemathlabelsep{\setlength{\labelsep}{\NCCPSiedmathlabelsep}}

% A flag which controls whether the IED lists automatically
% calculate \leftmargin from \NCCPSlabelindent, \labelwidth and \labelsep
% Useful if you want to specify your own \leftmargin
% This flag must be set (\NCCPSnocalcleftmargintrue or \NCCPSnocalcleftmarginfalse) 
% via the \NCCPSiedlistdecl or within the option of the IED list
% environments to have an effect.
\newif\ifNCCPSnocalcleftmargin
\NCCPSnocalcleftmarginfalse

% A flag which controls whether \NCCPSlabelindent is multiplied by
% the \NCCPSlabelindentfactor for each list level.
% This flag must be set via the \NCCPSiedlistdecl or within the option 
% of the IED list environments to have an effect.
\newif\ifNCCPSnolabelindentfactor
\NCCPSnolabelindentfactorfalse


% internal variable to indicate type of IED label
% justification
% 0 - left; 1 - center; 2 - right
\def\@NCCPSiedjustify{0}


% commands to allow the user to control IED
% label justifications. Use these commands within
% the IED environment option or in the \NCCPSiedlistdecl
% Note that changing the normal list justifications
% is nonstandard and the NCCPS may not like it if you do so!
% I include these commands as they may be helpful to
% those who are using these enhanced list controls for
% other non-NCCPS related LaTeX work.
% itemize and enumerate automatically default to right
% justification, description defaults to left.
\def\NCCPSiedlabeljustifyl{\def\@NCCPSiedjustify{0}}%left
\def\NCCPSiedlabeljustifyc{\def\@NCCPSiedjustify{1}}%center
\def\NCCPSiedlabeljustifyr{\def\@NCCPSiedjustify{2}}%right




% commands to save to and restore from the list parameter copies
% this allows us to set all the list parameters within
% the list_decl and prevent \list (and its \@list) 
% from overriding any of our parameters
% V1.6 use \edefs instead of dimen's to conserve dimen registers
% Note controlled spacing here, shield end of lines with %
\def\@NCCPSsavelistparams{\edef\@NCCPSiedtopsep{\the\topsep}%
\edef\@NCCPSiedlabelwidth{\the\labelwidth}%
\edef\@NCCPSiedlabelsep{\the\labelsep}%
\edef\@NCCPSiedleftmargin{\the\leftmargin}%
\edef\@NCCPSiedpartopsep{\the\partopsep}%
\edef\@NCCPSiedparsep{\the\parsep}%
\edef\@NCCPSieditemsep{\the\itemsep}%
\edef\@NCCPSiedrightmargin{\the\rightmargin}%
\edef\@NCCPSiedlistparindent{\the\listparindent}%
\edef\@NCCPSieditemindent{\the\itemindent}}

% Note controlled spacing here
\def\@NCCPSrestorelistparams{\topsep\@NCCPSiedtopsep\relax%
\labelwidth\@NCCPSiedlabelwidth\relax%
\labelsep\@NCCPSiedlabelsep\relax%
\leftmargin\@NCCPSiedleftmargin\relax%
\partopsep\@NCCPSiedpartopsep\relax%
\parsep\@NCCPSiedparsep\relax%
\itemsep\@NCCPSieditemsep\relax%
\rightmargin\@NCCPSiedrightmargin\relax%
\listparindent\@NCCPSiedlistparindent\relax%
\itemindent\@NCCPSieditemindent\relax}


% v1.6b provide original LaTeX IED list environments
% note that latex.ltx defines \itemize and \enumerate, but not \description
% which must be created by the base classes
% save original LaTeX itemize and enumerate
\let\LaTeXitemize\itemize
\let\endLaTeXitemize\enditemize
\let\LaTeXenumerate\enumerate
\let\endLaTeXenumerate\endenumerate

% provide original LaTeX description environment from article.cls
\newenvironment{LaTeXdescription}
               {\list{}{\labelwidth\z@ \itemindent-\leftmargin
                        \let\makelabel\descriptionlabel}}
               {\endlist}
\newcommand*\descriptionlabel[1]{\hspace\labelsep
                                 \normalfont\bfseries #1}


% override LaTeX's default IED lists
\def\itemize{\@NCCPSitemize}
\def\enditemize{\@endNCCPSitemize}
\def\enumerate{\@NCCPSenumerate}
\def\endenumerate{\@endNCCPSenumerate}
\def\description{\@NCCPSdescription}
\def\enddescription{\@endNCCPSdescription}

% provide the user with aliases - may help those using packages that
% override itemize, enumerate, or description
\def\NCCPSitemize{\@NCCPSitemize}
\def\endNCCPSitemize{\@endNCCPSitemize}
\def\NCCPSenumerate{\@NCCPSenumerate}
\def\endNCCPSenumerate{\@endNCCPSenumerate}
\def\NCCPSdescription{\@NCCPSdescription}
\def\endNCCPSdescription{\@endNCCPSdescription}


% V1.6 we want to keep the NCCPS IED list definitions as our own internal
% commands so they are protected against redefinition
\def\@NCCPSitemize{\@ifnextchar[{\@@NCCPSitemize}{\@@NCCPSitemize[\relax]}}
\def\@NCCPSenumerate{\@ifnextchar[{\@@NCCPSenumerate}{\@@NCCPSenumerate[\relax]}}
\def\@NCCPSdescription{\@ifnextchar[{\@@NCCPSdescription}{\@@NCCPSdescription[\relax]}}
\def\@endNCCPSitemize{\endlist}
\def\@endNCCPSenumerate{\endlist}
\def\@endNCCPSdescription{\endlist}


% DO NOT ALLOW BLANK LINES TO BE IN THESE IED ENVIRONMENTS
% AS THIS WILL FORCE NEW PARAGRAPHS AFTER THE IED LISTS
% NCCPS itemized list MDS 1/2001
% Note controlled spacing here, shield end of lines with %
\def\@@NCCPSitemize[#1]{%
                \ifnum\@itemdepth>3\relax\@toodeep\else%
                \ifnum\@listdepth>5\relax\@toodeep\else%
                \advance\@itemdepth\@ne%
                \edef\@itemitem{labelitem\romannumeral\the\@itemdepth}%
                % get the NCCPSlabelindentfactor for this level
                \advance\@listdepth\@ne% we need to know what the level WILL be
                \edef\NCCPSlabelindentfactor{\csname NCCPSlabelindentfactor\romannumeral\the\@listdepth\endcsname}%
                \advance\@listdepth-\@ne% undo our increment
                \def\@NCCPSiedjustify{2}% right justified labels are default
                % set other defaults
                \NCCPSnocalcleftmarginfalse%
                \NCCPSnolabelindentfactorfalse%
                \topsep\NCCPSiedtopsep%
                \NCCPSlabelindent\NCCPSilabelindent%
                \labelsep\NCCPSiednormlabelsep%
                \partopsep 0ex%
                \parsep 0ex%
                \itemsep 0ex%
                \rightmargin 0em%
                \listparindent 0em%
                \itemindent 0em%
                % calculate the label width
                % the user can override this later if
                % they specified a \labelwidth
                \settowidth{\labelwidth}{\csname labelitem\romannumeral\the\@itemdepth\endcsname}%
                \@NCCPSsavelistparams% save our list parameters
                \list{\csname\@itemitem\endcsname}{%
                \@NCCPSrestorelistparams% override any list{} changes
                                       % to our globals
                \let\makelabel\@NCCPSiedmakelabel% v1.6b setup \makelabel
                \NCCPSiedlistdecl% let user alter parameters
                #1\relax%
                % If the user has requested not to use the
                % NCCPSlabelindent factor, don't revise \NCCPSlabelindent
                \ifNCCPSnolabelindentfactor\relax%
                \else\NCCPSlabelindent=\NCCPSlabelindentfactor\NCCPSlabelindent%
                \fi%
                % Unless the user has requested otherwise,
                % calculate our left margin based
                % on \NCCPSlabelindent, \labelwidth and
                % \labelsep
                \ifNCCPSnocalcleftmargin\relax%
                \else\NCCPScalcleftmargin{\NCCPSlabelindent}%
                \fi}\fi\fi}%


% DO NOT ALLOW BLANK LINES TO BE IN THESE IED ENVIRONMENTS
% AS THIS WILL FORCE NEW PARAGRAPHS AFTER THE IED LISTS
% NCCPS enumerate list MDS 1/2001
% Note controlled spacing here, shield end of lines with %
\def\@@NCCPSenumerate[#1]{%
                \ifnum\@enumdepth>3\relax\@toodeep\else%
                \ifnum\@listdepth>5\relax\@toodeep\else%
                \advance\@enumdepth\@ne%
                \edef\@enumctr{enum\romannumeral\the\@enumdepth}%
                % get the NCCPSlabelindentfactor for this level
                \advance\@listdepth\@ne% we need to know what the level WILL be
                \edef\NCCPSlabelindentfactor{\csname NCCPSlabelindentfactor\romannumeral\the\@listdepth\endcsname}%
                \advance\@listdepth-\@ne% undo our increment
                \def\@NCCPSiedjustify{2}% right justified labels are default
                % set other defaults
                \NCCPSnocalcleftmarginfalse%
                \NCCPSnolabelindentfactorfalse%
                \topsep\NCCPSiedtopsep%
                \NCCPSlabelindent\NCCPSelabelindent%
                \labelsep\NCCPSiednormlabelsep%
                \partopsep 0ex%
                \parsep 0ex%
                \itemsep 0ex%
                \rightmargin 0em%
                \listparindent 0em%
                \itemindent 0em%
                % calculate the label width
                % We'll set it to the width suitable for all labels using
                % normalfont 1) to 9)
                % The user can override this later
                \settowidth{\labelwidth}{9)}%
                \@NCCPSsavelistparams% save our list parameters
                \list{\csname label\@enumctr\endcsname}{\usecounter{\@enumctr}%
                \@NCCPSrestorelistparams% override any list{} changes
                                       % to our globals
                \let\makelabel\@NCCPSiedmakelabel% v1.6b setup \makelabel
                \NCCPSiedlistdecl% let user alter parameters 
                #1\relax%
                % If the user has requested not to use the
                % NCCPSlabelindent factor, don't revise \NCCPSlabelindent
                \ifNCCPSnolabelindentfactor\relax%
                \else\NCCPSlabelindent=\NCCPSlabelindentfactor\NCCPSlabelindent%
                \fi%
                % Unless the user has requested otherwise,
                % calculate our left margin based
                % on \NCCPSlabelindent, \labelwidth and
                % \labelsep
                \ifNCCPSnocalcleftmargin\relax%
                \else\NCCPScalcleftmargin{\NCCPSlabelindent}%
                \fi}\fi\fi}%


% DO NOT ALLOW BLANK LINES TO BE IN THESE IED ENVIRONMENTS
% AS THIS WILL FORCE NEW PARAGRAPHS AFTER THE IED LISTS
% NCCPS description list MDS 1/2001
% Note controlled spacing here, shield end of lines with %
\def\@@NCCPSdescription[#1]{%
                \ifnum\@listdepth>5\relax\@toodeep\else%
                % get the NCCPSlabelindentfactor for this level
                \advance\@listdepth\@ne% we need to know what the level WILL be
                \edef\NCCPSlabelindentfactor{\csname NCCPSlabelindentfactor\romannumeral\the\@listdepth\endcsname}%
                \advance\@listdepth-\@ne% undo our increment
                \def\@NCCPSiedjustify{0}% left justified labels are default
                % set other defaults
                \NCCPSnocalcleftmarginfalse%
                \NCCPSnolabelindentfactorfalse%
                \topsep\NCCPSiedtopsep% 
                \NCCPSlabelindent\NCCPSdlabelindent%
                % assume normal labelsep
                \labelsep\NCCPSiednormlabelsep%
                \partopsep 0ex%
                \parsep 0ex%
                \itemsep 0ex%
                \rightmargin 0em%
                \listparindent 0em%
                \itemindent 0em%
                % Bogus label width in case the user forgets
                % to set it.
                % TIP: If you want to see what a variable's width is you
                % can use the TeX command \showthe\width-variable to 
                % display it on the screen during compilation 
                % (This might be helpful to know when you need to find out
                % which label is the widest)
                \settowidth{\labelwidth}{Hello}%
                \@NCCPSsavelistparams% save our list parameters
                \list{}{\@NCCPSrestorelistparams% override any list{} changes
                                               % to our globals
                \let\makelabel\@NCCPSiedmakelabel% v1.6b setup \makelabel
                \NCCPSiedlistdecl% let user alter parameters 
                #1\relax%
                % If the user has requested not to use the
                % labelindent factor, don't revise \NCCPSlabelindent
                \ifNCCPSnolabelindentfactor\relax%
                \else\NCCPSlabelindent=\NCCPSlabelindentfactor\NCCPSlabelindent%
                \fi%
                % Unless the user has requested otherwise,
                % calculate our left margin based
                % on \NCCPSlabelindent, \labelwidth and
                % \labelsep
                \ifNCCPSnocalcleftmargin\relax%
                \else\NCCPScalcleftmargin{\NCCPSlabelindent}\relax%
                \fi}\fi}

% v1.6b we use one makelabel that does justification as needed.
\def\@NCCPSiedmakelabel#1{\relax\if\@NCCPSiedjustify 0\relax
\makebox[\labelwidth][l]{\normalfont #1}\else
\if\@NCCPSiedjustify 1\relax
\makebox[\labelwidth][c]{\normalfont #1}\else
\makebox[\labelwidth][r]{\normalfont #1}\fi\fi}


% compsoc uses a larger value for the normal labelsep
% and also extra spacing above and below each list
\ifCLASSOPTIONcompsoc
  \NCCPSiednormlabelsep 1.2em
  \NCCPSiedtopsep 6pt plus 3pt minus 3pt
\fi


% VERSE and QUOTE
% V1.7 define environments with newenvironment
\newenvironment{verse}{\let\\=\@centercr
    \list{}{\itemsep\z@ \itemindent -1.5em \listparindent \itemindent
    \rightmargin\leftmargin\advance\leftmargin 1.5em}\item\relax}
    {\endlist}
\newenvironment{quotation}{\list{}{\listparindent 1.5em \itemindent\listparindent
    \rightmargin\leftmargin \parsep 0pt plus 1pt}\item\relax}
    {\endlist}
\newenvironment{quote}{\list{}{\rightmargin\leftmargin}\item\relax}
    {\endlist}


% \titlepage
% provided only for backward compatibility. \maketitle is the correct
% way to create the title page. 
\def\titlepage{\@restonecolfalse\if@twocolumn\@restonecoltrue\onecolumn
    \else \newpage \fi \thispagestyle{empty}\c@page\z@}
\def\endtitlepage{\if@restonecol\twocolumn \else \newpage \fi}

% standard values from article.cls
\arraycolsep     5pt
\arrayrulewidth .4pt
\doublerulesep   2pt

\tabcolsep       6pt
\tabbingsep      0.5em


%% FOOTNOTES
%
%\skip\footins 10pt plus 4pt minus 2pt
% V1.6 respond to changes in font size
% space added above the footnotes (if present)
\skip\footins 0.9\baselineskip  plus 0.4\baselineskip  minus 0.2\baselineskip

% V1.6, we need to make \footnotesep responsive to changes
% in \baselineskip or strange spacings will result when in
% draft mode. Here is a little LaTeX secret - \footnotesep
% determines the height of an invisible strut that is placed
% *above* the baseline of footnotes after the first. Since
% LaTeX considers the space for characters to be 0.7\baselineskip
% above the baseline and 0.3\baselineskip below it, we need to
% use 0.7\baselineskip as a \footnotesep to maintain equal spacing
% between all the lines of the footnotes. The NCCPS often uses a tad
% more, so use 0.8\baselineskip. This slightly larger value also helps
% the text to clear the footnote marks. Note that \thanks in NCCPS
% uses its own value of \footnotesep which is set in \maketitle.
{\footnotesize
\global\footnotesep 0.8\baselineskip}


\skip\@mpfootins = \skip\footins
\fboxsep = 3pt
\fboxrule = .4pt
% V1.6 use 1em, then use LaTeX2e's \@makefnmark
% Note that the NCCPS normally *left* aligns the footnote marks, so we don't need
% box resizing tricks here.
\long\def\@makefntext#1{\parindent 1em\indent\hbox{\@makefnmark}#1}% V1.6 use 1em
% V1.7 compsoc does not use superscipts for footnote marks
\ifCLASSOPTIONcompsoc
\def\@NCCPScompsocmakefnmark{\hbox{\normalfont\@thefnmark.\ }}
\long\def\@makefntext#1{\parindent 1em\indent\hbox{\@NCCPScompsocmakefnmark}#1}
\fi

% The NCCPS does not use footnote rules
\def\footnoterule{}

% V1.7 for compsoc, the NCCPS uses a footnote rule only for \thanks. We devise a "one-shot"
% system to implement this.
\newif\if@NCCPSenableoneshotfootnoterule
\@NCCPSenableoneshotfootnoterulefalse
\ifCLASSOPTIONcompsoc
\def\footnoterule{\relax\if@NCCPSenableoneshotfootnoterule
\kern-5pt
\hbox to \columnwidth{\hfill\vrule width 0.5\columnwidth height 0.4pt\hfill}
\kern4.6pt
\global\@NCCPSenableoneshotfootnoterulefalse
\else
\relax
\fi}
\fi

% V1.6 do not allow LaTeX to break a footnote across multiple pages
\interfootnotelinepenalty=10000

% V1.6 discourage breaks within equations
% Note that amsmath normally sets this to 10000,
% but LaTeX2e normally uses 100.
\interdisplaylinepenalty=2500

% default allows section depth up to /paragraph
\setcounter{secnumdepth}{4}

% technotes do not allow /paragraph
\ifCLASSOPTIONtechnote
   \setcounter{secnumdepth}{3}
\fi
% neither do compsoc conferences
\@NCCPScompsocconfonly{\setcounter{secnumdepth}{3}}


\newcounter{section}
\newcounter{subsection}[section]
\newcounter{subsubsection}[subsection]
\newcounter{paragraph}[subsubsection]

% used only by NCCPS's NCCPSeqnarray as other packages may
% have their own, different, implementations
\newcounter{NCCPSsubequation}[equation]

% as shown when called by user from \ref, \label and in table of contents
\def\theequation{\arabic{equation}}                          % 1
\def\theNCCPSsubequation{\theequation\alph{NCCPSsubequation}}  % 1a (used only by NCCPS's NCCPSeqnarray)
\ifCLASSOPTIONcompsoc
% compsoc is all arabic
\def\thesection{\arabic{section}}                
\def\thesubsection{\thesection.\arabic{subsection}}
\def\thesubsubsection{\thesubsection.\arabic{subsubsection}}
\def\theparagraph{\thesubsubsection.\arabic{paragraph}}
\else
\def\thesection{\Roman{section}}                             % I
% V1.7, \mbox prevents breaks around - 
\def\thesubsection{\mbox{\thesection-\Alph{subsection}}}     % I-A
% V1.7 use I-A1 format used by the NCCPS rather than I-A.1
\def\thesubsubsection{\thesubsection\arabic{subsubsection}}  % I-A1
\def\theparagraph{\thesubsubsection\alph{paragraph}}         % I-A1a
\fi

% From Heiko Oberdiek. Because of the \mbox in \thesubsection, we need to
% tell hyperref to disable the \mbox command when making PDF bookmarks.
% This done already with hyperref.sty version 6.74o and later, but
% it will not hurt to do it here again for users of older versions.
\@ifundefined{pdfstringdefPreHook}{\let\pdfstringdefPreHook\@empty}{}%
\g@addto@macro\pdfstringdefPreHook{\let\mbox\relax}


% Main text forms (how shown in main text headings)
% V1.6, using \thesection in \thesectiondis allows changes
% in the former to automatically appear in the latter
\ifCLASSOPTIONcompsoc
  \ifCLASSOPTIONconference% compsoc conference
    \def\thesectiondis{\thesection.}
    \def\thesubsectiondis{\thesectiondis\arabic{subsection}.}
    \def\thesubsubsectiondis{\thesubsectiondis\arabic{subsubsection}.}
    \def\theparagraphdis{\thesubsubsectiondis\arabic{paragraph}.}
  \else% compsoc not conferencs
    \def\thesectiondis{\thesection}
    \def\thesubsectiondis{\thesectiondis.\arabic{subsection}}
    \def\thesubsubsectiondis{\thesubsectiondis.\arabic{subsubsection}}
    \def\theparagraphdis{\thesubsubsectiondis.\arabic{paragraph}}
  \fi
\else% not compsoc
  \def\thesectiondis{\thesection.}                   % I.
  \def\thesubsectiondis{\Alph{subsection}.}          % B.
  \def\thesubsubsectiondis{\arabic{subsubsection})}  % 3)
  \def\theparagraphdis{\alph{paragraph})}            % d)
\fi

% just like LaTeX2e's \@eqnnum
\def\theequationdis{{\normalfont \normalcolor (\theequation)}}% (1)
% NCCPSsubequation used only by NCCPS's NCCPSeqnarray
\def\theNCCPSsubequationdis{{\normalfont \normalcolor (\theNCCPSsubequation)}}% (1a)
% redirect LaTeX2e's equation number display and all that depend on
% it, through NCCPS's \theequationdis
\def\@eqnnum{\theequationdis}



% V1.7 provide string macros as article.cls does
\def\contentsname{Contents}
\def\listfigurename{List of Figures}
\def\listtablename{List of Tables}
\def\refname{References}
\def\indexname{Index}
\def\figurename{Fig.}
\def\tablename{TABLE}
\@NCCPScompsocconfonly{\def\figurename{Figure}}
\def\partname{Part}
\def\appendixname{Appendix}
\def\abstractname{Abstract}
% NCCPS specific names
\def\NCCPSkeywordsname{Keywords}
\def\NCCPSproofname{Proof}


% LIST OF FIGURES AND TABLES AND TABLE OF CONTENTS
%
\def\@pnumwidth{1.55em}
\def\@tocrmarg{2.55em}
\def\@dotsep{4.5}
\setcounter{tocdepth}{3}

% adjusted some spacings here so that section numbers will not easily 
% collide with the section titles. 
% VIII; VIII-A; and VIII-A.1 are usually the worst offenders.
% MDS 1/2001
\def\tableofcontents{\section*{\contentsname}\@starttoc{toc}}
\def\l@section#1#2{\addpenalty{\@secpenalty}\addvspace{1.0em plus 1pt}%
    \@tempdima 2.75em \begingroup \parindent \z@ \rightskip \@pnumwidth%
    \parfillskip-\@pnumwidth {\bfseries\leavevmode #1}\hfil\hbox to\@pnumwidth{\hss #2}\par%
    \endgroup}
% argument format #1:level, #2:labelindent,#3:labelsep
\def\l@subsection{\@dottedtocline{2}{2.75em}{3.75em}}
\def\l@subsubsection{\@dottedtocline{3}{6.5em}{4.5em}}
% must provide \l@ defs for ALL sublevels EVEN if tocdepth
% is such as they will not appear in the table of contents
% these defs are how TOC knows what level these things are!
\def\l@paragraph{\@dottedtocline{4}{6.5em}{5.5em}}
\def\l@subparagraph{\@dottedtocline{5}{6.5em}{6.5em}}
\def\listoffigures{\section*{\listfigurename}\@starttoc{lof}}
\def\l@figure{\@dottedtocline{1}{0em}{2.75em}}
\def\listoftables{\section*{\listtablename}\@starttoc{lot}}
\let\l@table\l@figure


% Definitions for floats
%
% Normal Floats
% V1.8 floatsep et al. revised down by 0.15\baselineskip
% to account for the sideeffects of \topskip compensation
\floatsep 0.85\baselineskip plus  0.2\baselineskip minus  0.2\baselineskip
\textfloatsep 1.55\baselineskip plus  0.2\baselineskip minus  0.4\baselineskip
\@fptop 0pt plus 1fil
\@fpsep 0.75\baselineskip plus 2fil
\@fpbot 0pt plus 1fil
\def\topfraction{0.9}
\def\bottomfraction{0.4}
\def\floatpagefraction{0.8}
% V1.7, let top floats approach 90% of page
\def\textfraction{0.1}

% Double Column Floats
\dblfloatsep 0.85\baselineskip plus  0.2\baselineskip minus  0.2\baselineskip

\dbltextfloatsep 1.55\baselineskip plus  0.2\baselineskip minus  0.4\baselineskip
% Note that it would be nice if the rubber here actually worked in LaTeX2e.
% There is a long standing limitation in LaTeX, first discovered (to the best
% of my knowledge) by Alan Jeffrey in 1992. LaTeX ignores the stretchable
% portion of \dbltextfloatsep, and as a result, double column figures can and
% do result in an non-integer number of lines in the main text columns with
% underfull vbox errors as a consequence. A post to comp.text.tex
% by Donald Arseneau confirms that this had not yet been fixed in 1998.
% NCCPS V1.6 will fix this problem for you in the titles, but it doesn't
% protect you from other double floats. Happy vspace'ing.

\@dblfptop 0pt plus 1fil
\@dblfpsep 0.75\baselineskip plus 2fil
\@dblfpbot 0pt plus 1fil
\def\dbltopfraction{0.8}
\def\dblfloatpagefraction{0.8}
\setcounter{dbltopnumber}{4}

\intextsep 0.85\baselineskip plus 0.2\baselineskip minus  0.2\baselineskip
\setcounter{topnumber}{2}
\setcounter{bottomnumber}{2}
\setcounter{totalnumber}{4}



% article class provides these, we should too.
\newlength\abovecaptionskip
\newlength\belowcaptionskip
% but only \abovecaptionskip is used above figure captions and *below* table
% captions
\setlength\abovecaptionskip{0.5\baselineskip}
% compsoc journals are a little more generous
\ifCLASSOPTIONcompsoc\ifCLASSOPTIONjournal
  \setlength\abovecaptionskip{0.75\baselineskip}
\fi\fi
\setlength\belowcaptionskip{0pt}
% V1.6 create hooks in case the caption spacing ever needs to be
% overridden by a user
\def\@NCCPSfigurecaptionsepspace{\vskip\abovecaptionskip\relax}%
\def\@NCCPStablecaptionsepspace{\vskip\abovecaptionskip\relax}%


% 1.6b revise caption system so that \@makecaption uses two arguments
% as with LaTeX2e. Otherwise, there will be problems when using hyperref.
\def\@NCCPStablestring{table}


% V1.8 compensate for \topskip so top of top figures align with tops of the first lines of main text
% here we calculate a space equal to the amount \topskip exceeds the main text height
% we hook in at \@floatboxreset
\def\@NCCPSfiguretopskipspace{\ifdim\prevdepth=-1000pt\relax
\setlength{\@NCCPStmpdimenA}{1\topskip}\relax
\addtolength{\@NCCPStmpdimenA}{-0.7\@NCCPSnormalsizeunitybaselineskip}\relax
\vspace*{\@NCCPStmpdimenA}\fi}
% V1.8 compensate for \topskip at the top of top tables so caption text is on main text baseline
% use a strut set on the caption baseline within \@makecaption
\def\@NCCPStabletopskipstrut{\ifdim\prevdepth=-1000pt\rule{0pt}{\topskip}\fi}
% the \ifdim\prevdepth checks are always expected to be true for NCCPS style float caption ordering
% because top of figure content and top of captions in tables is the first thing on the vertical
% list of these floats
% thanks to Donald Arseneau for his 2000/11/11 post "Re: caption hacking" with info on this topic.


\ifCLASSOPTIONcompsoc
% V1.7 compsoc \@makecaption
\ifCLASSOPTIONconference% compsoc conference
\long\def\@makecaption#1#2{%
% test if is a for a figure or table
\ifx\@captype\@NCCPStablestring%
% if a table, do table caption
\footnotesize\bgroup\par\centering\@NCCPStabletopskipstrut{\normalfont\footnotesize {#1.}\nobreakspace\scshape #2}\par\addvspace{0.5\baselineskip}\egroup%
\@NCCPStablecaptionsepspace
% if not a table, format it as a figure
\else
\@NCCPSfigurecaptionsepspace
\setbox\@tempboxa\hbox{\normalfont\footnotesize {#1.}\nobreakspace #2}%
\ifdim \wd\@tempboxa >\hsize%
% if caption is longer than a line, let it wrap around
\setbox\@tempboxa\hbox{\normalfont\footnotesize {#1.}\nobreakspace}%
\parbox[t]{\hsize}{\normalfont\footnotesize \noindent\unhbox\@tempboxa#2}%
% if caption is shorter than a line, center
\else%
\hbox to\hsize{\normalfont\footnotesize\hfil\box\@tempboxa\hfil}%
\fi\fi}
%
\else% nonconference compsoc
\long\def\@makecaption#1#2{%
% test if is a for a figure or table
\ifx\@captype\@NCCPStablestring%
% if a table, do table caption
\footnotesize\bgroup\par\centering\@NCCPStabletopskipstrut{\normalfont\sffamily\footnotesize #1}\\{\normalfont\sffamily\footnotesize #2}\par\addvspace{0.5\baselineskip}\egroup%
\@NCCPStablecaptionsepspace
% if not a table, format it as a figure
\else
\@NCCPSfigurecaptionsepspace
\setbox\@tempboxa\hbox{\normalfont\sffamily\footnotesize {#1.}\nobreakspace #2}%
\ifdim \wd\@tempboxa >\hsize%
% if caption is longer than a line, let it wrap around
\setbox\@tempboxa\hbox{\normalfont\sffamily\footnotesize {#1.}\nobreakspace}%
\parbox[t]{\hsize}{\normalfont\sffamily\footnotesize \noindent\unhbox\@tempboxa#2}%
% if caption is shorter than a line, left justify
\else%
\hbox to\hsize{\normalfont\sffamily\footnotesize\box\@tempboxa\hfil}%
\fi\fi}
\fi
%
\else% traditional noncompsoc \@makecaption
\long\def\@makecaption#1#2{%
% test if is a for a figure or table
\ifx\@captype\@NCCPStablestring%
% if a table, do table caption
\footnotesize\bgroup\par\centering\@NCCPStabletopskipstrut{\normalfont\footnotesize #1}\\{\normalfont\footnotesize\scshape #2}\par\addvspace{0.5\baselineskip}\egroup%
\@NCCPStablecaptionsepspace
% if not a table, format it as a figure
\else
\@NCCPSfigurecaptionsepspace
% 3/2001 use footnotesize, not small; use two nonbreaking spaces, not one
\setbox\@tempboxa\hbox{\normalfont\footnotesize {#1.}\nobreakspace\nobreakspace #2}%
\ifdim \wd\@tempboxa >\hsize%
% if caption is longer than a line, let it wrap around
\setbox\@tempboxa\hbox{\normalfont\footnotesize {#1.}\nobreakspace\nobreakspace}%
\parbox[t]{\hsize}{\normalfont\footnotesize\noindent\unhbox\@tempboxa#2}%
% if caption is shorter than a line, center if conference, left justify otherwise
\else%
\ifCLASSOPTIONconference \hbox to\hsize{\normalfont\footnotesize\hfil\box\@tempboxa\hfil}%
\else \hbox to\hsize{\normalfont\footnotesize\box\@tempboxa\hfil}%
\fi\fi\fi}
\fi



% V1.7 disable captions class option, do so in a way that retains operation of \label
% within \caption
\ifCLASSOPTIONcaptionsoff
\long\def\@makecaption#1#2{\vspace*{2em}\footnotesize\bgroup\par\addvspace{0.5\baselineskip}\centering{\footnotesize #1}\par\addvspace{0.5\baselineskip}\egroup%
\let\@NCCPStemporiglabeldefsave\label
\let\@NCCPStemplabelargsave\relax
\def\label##1{\gdef\@NCCPStemplabelargsave{##1}}%
\setbox\@tempboxa\hbox{#2}%
\let\label\@NCCPStemporiglabeldefsave
\ifx\@NCCPStemplabelargsave\relax\else\label{\@NCCPStemplabelargsave}\fi}
\fi


% V1.7 define end environments with \def not \let so as to work OK with
% preview-latex
\newcounter{figure}
\def\thefigure{\@arabic\c@figure}
\def\fps@figure{tbp}
\def\ftype@figure{1}
\def\ext@figure{lof}
\def\fnum@figure{\figurename\nobreakspace\thefigure}
% V1.8 within figures add \@NCCPSfiguretopskipspace compensation to LaTeX2e's \@floatboxreset
\def\figure{\def\@floatboxreset{\reset@font\normalsize\@setminipage\@NCCPSfiguretopskipspace}\@float{figure}}
\def\endfigure{\end@float}
% V1.8 also add \@NCCPSfiguretopskipspace compensation to \figure*
\@namedef{figure*}{\def\@floatboxreset{\reset@font\normalsize\@setminipage\@NCCPSfiguretopskipspace}\@dblfloat{figure}}
\@namedef{endfigure*}{\end@dblfloat}

\newcounter{table}
\ifCLASSOPTIONcompsoc
\def\thetable{\arabic{table}}
\else
\def\thetable{\@Roman\c@table}
\fi
\def\fps@table{tbp}
\def\ftype@table{2}
\def\ext@table{lot}
\def\fnum@table{\tablename\nobreakspace\thetable}
% V1.6 The NCCPS uses 8pt text for tables
% within tables alter LaTeX2e's \@floatboxreset to use \footnotesize
\def\table{\def\@floatboxreset{\reset@font\footnotesize\@setminipage}\@float{table}}
\def\endtable{\end@float}
% v1.6b double column tables need to default to footnotesize as well.
\@namedef{table*}{\def\@floatboxreset{\reset@font\footnotesize\@setminipage}\@dblfloat{table}}
\@namedef{endtable*}{\end@dblfloat}




%% -- Command Argument Scanning Support Functions --
%% V1.8a 

% usage: \@NCCPSstripouterbraces*{}
% \@NCCPSstripouterbraces fully expands its argument (which it then stores
% in \@NCCPSstripouterbracesarg) via \edef, then removes any outer enclosing
% braces, and finally stores the result in the macro
% \@NCCPSstrippedouterbraces.
%
% For example:
% \@NCCPSstripouterbraces{{{{ab}c}}}
% results in:
%
% \@NCCPSstripouterbracesarg ==> a macro containing {{{ab}c}}
% \@NCCPSstrippedouterbraces ==> a macro containing {ab}c
%
% the *-star form,\@NCCPSstripouterbraces*, does not expand the argument
% contents during processing
\def\@NCCPSstripouterbraces{\@ifstar{\let\@NCCPSstripouterbracesdef=\def\@@NCCPSstripouterbraces}{\let\@NCCPSstripouterbracesdef=\edef\@@NCCPSstripouterbraces}}

\def\@@NCCPSstripouterbraces#1{\@NCCPSstripouterbracesdef\@NCCPSstripouterbracesarg{#1}\relax
% If the macro is unchanged after being acquired as a single delimited
% argument, we know we have one sequence of tokens without any enclosing
% braces. Loop until this is true.
\loop
  \expandafter\@@@NCCPSstripouterbraces\@NCCPSstripouterbracesarg\@NCCPSgeneralsequenceDELIMITER
\ifx\@NCCPSstrippedouterbraces\@NCCPSstripouterbracesarg
\else
  \let\@NCCPSstripouterbracesarg\@NCCPSstrippedouterbraces
\repeat}

\def\@@@NCCPSstripouterbraces#1\@NCCPSgeneralsequenceDELIMITER{\def\@NCCPSstrippedouterbraces{#1}}



% usage: \@NCCPSextractgroup*{}
% \@NCCPSextractgroup fully expands its argument (which it then stores in
% \@NCCPSextractgrouparg) via \edef and then assigns the first "brace group"
% of tokens to the macro \@NCCPSextractedgroup.
% The remaining groups, if any, are stored in the macro
% \@NCCPSextractedgroupremain. If the argument does not contain the requisite
% groups, the respective macros will be defined to be empty.
% There is an asymmetry in that \@NCCPSextractedgroup is stripped of its first
% outer grouping while \@NCCPSextractedgroupremain retains even the outer
% grouping (if present) that originally identified it as a group.
%
% For example:
% \@NCCPSextractgroup{{{ab}}{c{de}}}
% results in:
%
% \@NCCPSextractgrouparg       ==> a macro containing {{ab}}{c{de}}
% \@NCCPSextractedgroup        ==> a macro containing {ab}
% \@NCCPSextractedgroupremain  ==> a macro containing {c{de}}
%
% The *-star form, \@NCCPSextractgroup*, does not expand its argument
% contents during processing.
\def\@NCCPSextractgroup{\@ifstar{\let\@NCCPSextractgroupdef=\def\@@NCCPSextractgroup}{\let\@NCCPSextractgroupdef=\edef\@@NCCPSextractgroup}}

\def\@@NCCPSextractgroup#1{\@NCCPSextractgroupdef\@NCCPSextractgrouparg{#1}\relax
% trap the case of an empty extracted group as this would cause problems with
% \@NCCPSextractgroupremain's argument acquisition
\ifx\@NCCPSextractgrouparg\@empty
  \def\@NCCPSextractedgroup{}\relax
  \def\@NCCPSextractedgroupremain{}\relax
\else
  % We have to use some dirty tricks here. We want to insert {} around
  % whatever remains after the first group so that TeX's argument scanner
  % will preserve any originally enclosing braces as well as provide an
  % empty argument to acquire even if there isn't a second group.
  % In this first of two dirty tricks, we put a } at the end of the structure
  % we are going to extract from. The \ifnum0=`{\fi keeps TeX happy to allow
  % what would otherwise be an unbalanced macro definition for
  % \@@NCCPSextractgroup to be acceptable to it.
  \ifnum0=`{\fi\expandafter\@NCCPSextractgroupremain\@NCCPSextractgrouparg}\relax
\fi}

% In the second part of the dirty tricks, we insert a leading { right after
% the first group is acquired, but before the remainder is. Again, the
% \ifnum0=`}\fi keeps TeX happy during definition time, but will disappear
% during run time.
\def\@NCCPSextractgroupremain#1{\def\@NCCPSextractedgroup{#1}\expandafter\@@NCCPSextractgroupremain\expandafter{\ifnum0=`}\fi}

\def\@@NCCPSextractgroupremain#1{\def\@NCCPSextractedgroupremain{#1}}



% \@NCCPSextracttoken relocated at top because margin setting commands rely on it



% usage: \@NCCPSextracttokengroups*{}
% \@NCCPSextracttokengroups fully expands its argument (which it then stores
% in \@NCCPSextracttokengroupsarg) and then assigns the first "brace group" of
% tokens (with the outermost braces removed) to the macro
% \@NCCPSextractedfirstgroup.
% The meaning of the first nonbrace (but including the empty group) token
% within this first group is assigned via \let to \@NCCPSextractedfirsttoken
% as well as stored in the macro \@NCCPSextractedfirsttokenmacro. If a first
% nonbrace token does not exist (or is an empty group), these will be \relax
% and empty, respectively.  Tokens that would otherwise be discarded during
% the acquisition of the first token in the first group are stored in
% \@NCCPSextractedfirsttokensdiscarded, however their original relative brace
% nesting depths are not guaranteed to be preserved.
% The first group within this first group is stored in the macro
% \@NCCPSextractedfirstfirstgroup.
% Likewise for the next group after the first: \@NCCPSextractednextgroup,
% \@NCCPSextractednextfirstgroup, \@NCCPSextractednextgroupfirsttoken,
% \@NCCPSextractednextgroupfirsttokenmacro, and 
% \@NCCPSextractednextfirsttokensdiscarded.
% All tokens/groups after the first group, including any enclosing braces,
% are stored in the macro \@NCCPSextractedafterfirstgroupremain which will
% be empty if none exist.
%
% For example:
% \@NCCPSextracttokengroups{{{ab}{cd}}{{ef}g}}
% will result in:
%
% \@NCCPSextracttokengroupsarg             ==> a macro containing {{ab}{cd}}{{ef}g}
% \@NCCPSextractedfirstgroup               ==> a macro containing {ab}{cd}
% \@NCCPSextractedafterfirstgroupremain    ==> a macro containing {{ef}g}
% \@NCCPSextractedfirsttoken               ==> the letter a
% \@NCCPSextractedfirsttokenmacro          ==> a macro containing a
% \@NCCPSextractedfirsttokensdiscarded     ==> a macro containing bcd
% \@NCCPSextractedfirstfirstgroup          ==> a macro containing ab
% \@NCCPSextractednextgroup                ==> a macro containing {ef}g
% \@NCCPSextractednextfirsttoken           ==> the letter e
% \@NCCPSextractednextfirsttokenmacro      ==> a macro containing e
% \@NCCPSextractednextfirsttokensdiscarded ==> a macro containing fg
% \@NCCPSextractednextfirstgroup           ==> a macro containing ef
%
% If given an empty argument, \@NCCPSextractedfirsttoken and
% \@NCCPSextractednextfirsttoken will be set to \relax
% and all the macros will be empty.
% the *-star form, \@NCCPSextracttokengroups*, does not expand its argument
% contents during processing.
%
% Depends on: \@NCCPSextractgroup, \@NCCPSextracttoken
\def\@NCCPSextracttokengroups{\@ifstar{\let\@NCCPSextracttokengroupsdef=\def\@@NCCPSextracttokengroups}{\let\@NCCPSextracttokengroupsdef=\edef\@@NCCPSextracttokengroups}}
\def\@@NCCPSextracttokengroups#1{\@NCCPSextracttokengroupsdef\@NCCPSextracttokengroupsarg{#1}\relax
% begin extraction, these functions are safe with empty arguments
% first group
\expandafter\@NCCPSextractgroup\expandafter*\expandafter{\@NCCPSextracttokengroupsarg}\relax
\let\@NCCPSextractedfirstgroup\@NCCPSextractedgroup
\let\@NCCPSextractedafterfirstgroupremain\@NCCPSextractedgroupremain
\expandafter\@NCCPSextracttoken\expandafter*\expandafter{\@NCCPSextractedfirstgroup}\relax
\let\@NCCPSextractedfirsttoken\@NCCPSextractedtoken
\let\@NCCPSextractedfirsttokenmacro\@NCCPSextractedtokenmacro
\let\@NCCPSextractedfirsttokensdiscarded\@NCCPSextractedtokensdiscarded
% first first group
\expandafter\@NCCPSextractgroup\expandafter*\expandafter{\@NCCPSextractedfirstgroup}\relax
\let\@NCCPSextractedfirstfirstgroup\@NCCPSextractedgroup
% next group
\expandafter\@NCCPSextractgroup\expandafter*\expandafter{\@NCCPSextractedafterfirstgroupremain}\relax
\let\@NCCPSextractednextgroup\@NCCPSextractedgroup
\expandafter\@NCCPSextracttoken\expandafter*\expandafter{\@NCCPSextractednextgroup}\relax
\let\@NCCPSextractednextfirsttoken\@NCCPSextractedtoken
\let\@NCCPSextractednextfirsttokenmacro\@NCCPSextractedtokenmacro
\let\@NCCPSextractednextfirsttokensdiscarded\@NCCPSextractedtokensdiscarded
% next first group
\expandafter\@NCCPSextractgroup\expandafter*\expandafter{\@NCCPSextractednextgroup}\relax
\let\@NCCPSextractednextfirstgroup\@NCCPSextractedgroup}


%% -- End of Command Argument Scanning Support Functions --




%%
%% START OF NCCPSeqnarray DEFINITIONS
%%
%% Inspired by the concepts, examples, and previous works of LaTeX 
%% coders and developers such as Donald Arseneau, Fred Bartlett, 
%% David Carlisle, Tony Liu, Frank Mittelbach, Piet van Oostrum, 
%% Roland Winkler and Mark Wooding.
%% I don't make the claim that my work here is even near their calibre. ;)


\newif\if@NCCPSeqnarrayboxnojot% flag to indicate if the environment was called as the star form
\@NCCPSeqnarrayboxnojotfalse

\newif\if@advanceNCCPSeqncolcnt% tracks if the environment should advance the col counter
% allows a way to make an \NCCPSeqnarraybox that can be used within an \NCCPSeqnarray
% used by NCCPSeqnarraymulticol so that it can work properly in both
\@advanceNCCPSeqncolcnttrue

\newcount\@NCCPSeqnnumcols % tracks how many NCCPSeqnarray cols are defined
\newcount\@NCCPSeqncolcnt  % tracks how many NCCPSeqnarray cols the user actually used


% The default math style used by the columns
\def\NCCPSeqnarraymathstyle{\displaystyle}
% The default text style used by the columns
% default to using the current font
\def\NCCPSeqnarraytextstyle{\relax}

% like the iedlistdecl but for \NCCPSeqnarray
\def\NCCPSeqnarraydecl{\relax}
\def\NCCPSeqnarrayboxdecl{\relax}



% V1.8 flags to indicate that equation numbering is to persist
\newif\if@NCCPSeqnumpersist%
\@NCCPSeqnumpersistfalse
\newif\if@NCCPSsubeqnumpersist%
\@NCCPSsubeqnumpersistfalse
%
% V1.8 flags to indicate if (sub)equation number of last line was preadvanced
\newif\if@NCCPSeqnumpreadv%
\@NCCPSeqnumpreadvfalse
\newif\if@NCCPSsubeqnumpreadv%
\@NCCPSsubeqnumpreadvfalse

\newcount\@NCCPSsubeqnnumrollback% saves previous value of NCCPSsubequation number in case we need to restore it

% \yesnumber is the opposite of \nonumber
% a novel concept with the same def as the equationarray package
% However, we give NCCPS versions too since some LaTeX packages such as 
% the MDWtools mathenv.sty redefine \nonumber to something else.
% This command is intended for use in non-NCCPSeqnarray math environments
\providecommand{\yesnumber}{\global\@eqnswtrue}


% NCCPSyes/nonumber 
% V1.8 add persistant * forms
% These commands can alter the type of equation an NCCPSeqnarray line is.
\def\NCCPSyesnumber{\@ifstar{\global\@NCCPSeqnumpersisttrue\global\@NCCPSsubeqnumpersistfalse\@NCCPSyesnumber}{\@NCCPSyesnumber}}

\def\@NCCPSyesnumber{\global\@eqnswtrue
\if@NCCPSeqnarrayISinner% alter counters and label only inside an NCCPSeqnarray
\ifnum\c@NCCPSsubequation>0\relax
   \stepcounter{equation}\setcounter{NCCPSsubequation}{0}\gdef\@currentlabel{\p@equation\theequation}\relax
   \gdef\@currentHref{\@NCCPStheHrefequation}% setup hyperref label
\fi
% even if we reached this eqn num via a preadv, it is legit now
\global\@NCCPSeqnumpreadvfalse\global\@NCCPSsubeqnumpreadvfalse
\fi}

\def\NCCPSnonumber{\@ifstar{\global\@NCCPSeqnumpersistfalse\global\@NCCPSsubeqnumpersistfalse\global\@eqnswfalse}{\global\@eqnswfalse}}


\def\NCCPSyessubnumber{\@ifstar{\global\@NCCPSsubeqnumpersisttrue\@NCCPSyessubnumber}{\@NCCPSyessubnumber}}
%
\def\@NCCPSyessubnumber{\if@NCCPSeqnarrayISinner% alter counters and label only inside an NCCPSeqnarray
  \ifnum\c@NCCPSsubequation>0\relax% if it already is a subequation, we are good to go as-is
  \else% if we are a regular equation we have to watch out for two cases
    \if@NCCPSeqnumpreadv% if this equation is the result of a preadvance, backout and bump the sub eqnnum
       \global\advance\c@equation\m@ne\global\c@NCCPSsubequation=\@NCCPSsubeqnnumrollback\addtocounter{NCCPSsubequation}{1}\relax
    \else% non-preadvanced equations just need initialization of their sub eqnnum
       \setcounter{NCCPSsubequation}{1}\relax
    \fi
  \fi% fi already is subequation
  \gdef\@currentlabel{\p@NCCPSsubequation\theNCCPSsubequation}\relax
  \gdef\@currentHref{\@NCCPStheHrefsubequation}% setup hyperref label
  \global\@NCCPSeqnumpreadvfalse\global\@NCCPSsubeqnumpreadvfalse% no longer a preadv anymore
  \global\@eqnswtrue
\fi}


\def\NCCPSnosubnumber{\@ifstar{\global\@NCCPSsubeqnumpersistfalse\@NCCPSnosubnumber}{\@NCCPSnosubnumber}}
%
\def\@NCCPSnosubnumber{\if@NCCPSeqnarrayISinner% alter counters and label only inside an NCCPSeqnarray
  \if@eqnsw % we do nothing unless we know we will display because we play with the counters here
    % if it currently is a subequation, bump up to the next equation number and turn off the subequation
    \ifnum\c@NCCPSsubequation>0\relax\addtocounter{equation}{1}\setcounter{NCCPSsubequation}{0}\relax
    \fi
    \global\@NCCPSeqnumpreadvfalse\global\@NCCPSsubeqnumpreadvfalse% no longer a preadv anymore
    \gdef\@currentlabel{\p@equation\theequation}\relax
    \gdef\@currentHref{\@NCCPStheHrefequation}% setup hyperref label
  \fi
\fi}



% allows users to "push away" equations that get too close to the equation numbers
\def\NCCPSeqnarraynumspace{\hphantom{\ifnum\c@NCCPSsubequation>0\relax\theNCCPSsubequationdis\else\theequationdis\fi}}

% provides a way to span multiple columns within NCCPSeqnarray environments
% will consider \if@advanceNCCPSeqncolcnt before globally advancing the
% column counter - so as to work within \NCCPSeqnarraybox
% usage: \NCCPSeqnarraymulticol{number cols. to span}{col type}{cell text}
\long\def\NCCPSeqnarraymulticol#1#2#3{\multispan{#1}\relax
% check if column is defined for the precolumn definition
% We have to be careful here because TeX scans for & even within an \iffalse
% where it does not expand macros. So, if we used only one \ifx and a #3
% appeared in the false branch and the user inserted another alignment
% structure that uses & in the \NCCPSeqnarraymulticol{}, TeX will not see that
% there is an inner alignment in the false branch yet still will see any &
% there and will think that they apply to the outer alignment resulting in an
% incomplete \ifx error.
% So, here we use separate checks for the pre and post parts in order to keep
% the #3 outside of all conditionals.
\relax\expandafter\ifx\csname @NCCPSeqnarraycolDEF#2\endcsname\@NCCPSeqnarraycolisdefined\relax
\csname @NCCPSeqnarraycolPRE#2\endcsname
\else% if not, error and use default type
\@NCCPSclspkgerror{Invalid column type "#2" in \string\NCCPSeqnarraymulticol.\MessageBreak
Using a default centering column instead}%
{You must define NCCPSeqnarray column types before use.}%
\csname @NCCPSeqnarraycolPRE@NCCPSdefault\endcsname
\fi
% The ten \relax are to help prevent misleading error messages in case a user
% accidently inserted a macro that tries to acquire additional arguments.
#3\relax\relax\relax\relax\relax\relax\relax\relax\relax\relax
% check if column is defined for the postcolumn definition
\expandafter\ifx\csname @NCCPSeqnarraycolDEF#2\endcsname\@NCCPSeqnarraycolisdefined\relax
\csname @NCCPSeqnarraycolPOST#2\endcsname
\else% if not, use the default type
\csname @NCCPSeqnarraycolPOST@NCCPSdefault\endcsname
\fi
% advance column counter only if the NCCPSeqnarray environment wants it
\if@advanceNCCPSeqncolcnt\global\advance\@NCCPSeqncolcnt by #1\relax\fi}

% like \omit, but maintains track of the column counter for \NCCPSeqnarray
\def\NCCPSeqnarrayomit{\omit\if@advanceNCCPSeqncolcnt\global\advance\@NCCPSeqncolcnt by 1\relax\fi}


% provides a way to define a letter referenced column type
% usage: \NCCPSeqnarraydefcol{col. type letter/name}{pre insertion text}{post insertion text}
\def\NCCPSeqnarraydefcol#1#2#3{\expandafter\def\csname @NCCPSeqnarraycolPRE#1\endcsname{#2}%
\expandafter\def\csname @NCCPSeqnarraycolPOST#1\endcsname{#3}%
\expandafter\def\csname @NCCPSeqnarraycolDEF#1\endcsname{1}}


% provides a way to define a numerically referenced inter-column glue types
% usage: \NCCPSeqnarraydefcolsep{col. glue number}{glue definition}
\def\NCCPSeqnarraydefcolsep#1#2{\expandafter\def\csname @NCCPSeqnarraycolSEP\romannumeral #1\endcsname{#2}%
\expandafter\def\csname @NCCPSeqnarraycolSEPDEF\romannumeral #1\endcsname{1}}


\def\@NCCPSeqnarraycolisdefined{1}% just a macro for 1, used for checking undefined column types


% expands and appends the given argument to the \@NCCPStmptoksA token list
% used to build up the \halign preamble
\def\@NCCPSappendtoksA#1{\edef\@@NCCPSappendtoksA{\@NCCPStmptoksA={\the\@NCCPStmptoksA #1}}%
\@@NCCPSappendtoksA}

% also appends to \@NCCPStmptoksA, but does not expand the argument
% uses \toks8 as a scratchpad register
\def\@NCCPSappendNOEXPANDtoksA#1{\toks8={#1}%
\edef\@@NCCPSappendNOEXPANDtoksA{\@NCCPStmptoksA={\the\@NCCPStmptoksA\the\toks8}}%
\@@NCCPSappendNOEXPANDtoksA}

% define some common column types for the user
% math
\NCCPSeqnarraydefcol{l}{$\NCCPSeqnarraymathstyle}{$\hfil}
\NCCPSeqnarraydefcol{c}{\hfil$\NCCPSeqnarraymathstyle}{$\hfil}
\NCCPSeqnarraydefcol{r}{\hfil$\NCCPSeqnarraymathstyle}{$}
\NCCPSeqnarraydefcol{L}{$\NCCPSeqnarraymathstyle{}}{{}$\hfil}
\NCCPSeqnarraydefcol{C}{\hfil$\NCCPSeqnarraymathstyle{}}{{}$\hfil}
\NCCPSeqnarraydefcol{R}{\hfil$\NCCPSeqnarraymathstyle{}}{{}$}
% text
\NCCPSeqnarraydefcol{s}{\NCCPSeqnarraytextstyle}{\hfil}
\NCCPSeqnarraydefcol{t}{\hfil\NCCPSeqnarraytextstyle}{\hfil}
\NCCPSeqnarraydefcol{u}{\hfil\NCCPSeqnarraytextstyle}{}

% vertical rules
\NCCPSeqnarraydefcol{v}{}{\vrule width\arrayrulewidth}
\NCCPSeqnarraydefcol{vv}{\vrule width\arrayrulewidth\hfil}{\hfil\vrule width\arrayrulewidth}
\NCCPSeqnarraydefcol{V}{}{\vrule width\arrayrulewidth\hskip\doublerulesep\vrule width\arrayrulewidth}
\NCCPSeqnarraydefcol{VV}{\vrule width\arrayrulewidth\hskip\doublerulesep\vrule width\arrayrulewidth\hfil}%
{\hfil\vrule width\arrayrulewidth\hskip\doublerulesep\vrule width\arrayrulewidth}

% horizontal rules
\NCCPSeqnarraydefcol{h}{}{\leaders\hrule height\arrayrulewidth\hfil}
\NCCPSeqnarraydefcol{H}{}{\leaders\vbox{\hrule width\arrayrulewidth\vskip\doublerulesep\hrule width\arrayrulewidth}\hfil}

% plain
\NCCPSeqnarraydefcol{x}{}{}
\NCCPSeqnarraydefcol{X}{$}{$}

% the default column type to use in the event a column type is not defined
\NCCPSeqnarraydefcol{@NCCPSdefault}{\hfil$\NCCPSeqnarraymathstyle}{$\hfil}


% a zero tabskip (used for "-" col types)
\def\@NCCPSeqnarraycolSEPzero{0pt plus 0pt minus 0pt}
% a centering tabskip (used for "+" col types)
\def\@NCCPSeqnarraycolSEPcenter{1000pt plus 0pt minus 1000pt}

% top level default tabskip glues for the start, end, and inter-column
% may be reset within environments not always at the top level, e.g., \NCCPSeqnarraybox
\edef\@NCCPSeqnarraycolSEPdefaultstart{\@NCCPSeqnarraycolSEPcenter}% default start glue
\edef\@NCCPSeqnarraycolSEPdefaultend{\@NCCPSeqnarraycolSEPcenter}% default end glue
\edef\@NCCPSeqnarraycolSEPdefaultmid{\@NCCPSeqnarraycolSEPzero}% default inter-column glue



% creates a vertical rule that extends from the bottom to the top a a cell
% Provided in case other packages redefine \vline some other way.
% usage: \NCCPSeqnarrayvrule[rule thickness]
% If no argument is provided, \arrayrulewidth will be used for the rule thickness. 
\newcommand\NCCPSeqnarrayvrule[1][\arrayrulewidth]{\vrule\@width#1\relax}

% creates a blank separator row
% usage: \NCCPSeqnarrayseprow[separation length][font size commands]
% default is \NCCPSeqnarrayseprow[0.25\normalbaselineskip][\relax]
% blank arguments inherit the default values
% uses \skip5 as a scratch register - calls \@NCCPSeqnarraystrutsize which uses more scratch registers
\def\NCCPSeqnarrayseprow{\relax\@ifnextchar[{\@NCCPSeqnarrayseprow}{\@NCCPSeqnarrayseprow[0.25\normalbaselineskip]}}
\def\@NCCPSeqnarrayseprow[#1]{\relax\@ifnextchar[{\@@NCCPSeqnarrayseprow[#1]}{\@@NCCPSeqnarrayseprow[#1][\relax]}}
\def\@@NCCPSeqnarrayseprow[#1][#2]{\def\@NCCPSeqnarrayseprowARGONE{#1}%
\ifx\@NCCPSeqnarrayseprowARGONE\@empty%
% get the skip value, based on the font commands
% use skip5 because \NCCPSeqnarraystrutsize uses \skip0, \skip2, \skip3
% assign within a bogus box to confine the font changes
{\setbox0=\hbox{#2\relax\global\skip5=0.25\normalbaselineskip}}%
\else%
{\setbox0=\hbox{#2\relax\global\skip5=#1}}%
\fi%
\@NCCPSeqnarrayhoptolastcolumn\NCCPSeqnarraystrutsize{\skip5}{0pt}[\relax]\relax}

% creates a blank separator row, but omits all the column templates
% usage: \NCCPSeqnarrayseprowcut[separation length][font size commands]
% default is \NCCPSeqnarrayseprowcut[0.25\normalbaselineskip][\relax]
% blank arguments inherit the default values
% uses \skip5 as a scratch register - calls \@NCCPSeqnarraystrutsize which uses more scratch registers
\def\NCCPSeqnarrayseprowcut{\multispan{\@NCCPSeqnnumcols}\relax% span all the cols
% advance column counter only if the NCCPSeqnarray environment wants it
\if@advanceNCCPSeqncolcnt\global\advance\@NCCPSeqncolcnt by \@NCCPSeqnnumcols\relax\fi%
\@ifnextchar[{\@NCCPSeqnarrayseprowcut}{\@NCCPSeqnarrayseprowcut[0.25\normalbaselineskip]}}
\def\@NCCPSeqnarrayseprowcut[#1]{\relax\@ifnextchar[{\@@NCCPSeqnarrayseprowcut[#1]}{\@@NCCPSeqnarrayseprowcut[#1][\relax]}}
\def\@@NCCPSeqnarrayseprowcut[#1][#2]{\def\@NCCPSeqnarrayseprowARGONE{#1}%
\ifx\@NCCPSeqnarrayseprowARGONE\@empty%
% get the skip value, based on the font commands
% use skip5 because \NCCPSeqnarraystrutsize uses \skip0, \skip2, \skip3
% assign within a bogus box to confine the font changes
{\setbox0=\hbox{#2\relax\global\skip5=0.25\normalbaselineskip}}%
\else%
{\setbox0=\hbox{#2\relax\global\skip5=#1}}%
\fi%
\NCCPSeqnarraystrutsize{\skip5}{0pt}[\relax]\relax}



% draws a single rule across all the columns optional
% argument determines the rule width, \arrayrulewidth is the default
% updates column counter as needed and turns off struts
% usage: \NCCPSeqnarrayrulerow[rule line thickness]
\def\NCCPSeqnarrayrulerow{\multispan{\@NCCPSeqnnumcols}\relax% span all the cols
% advance column counter only if the NCCPSeqnarray environment wants it
\if@advanceNCCPSeqncolcnt\global\advance\@NCCPSeqncolcnt by \@NCCPSeqnnumcols\relax\fi%
\@ifnextchar[{\@NCCPSeqnarrayrulerow}{\@NCCPSeqnarrayrulerow[\arrayrulewidth]}}
\def\@NCCPSeqnarrayrulerow[#1]{\leaders\hrule height#1\hfil\relax% put in our rule 
% turn off any struts
\NCCPSeqnarraystrutsize{0pt}{0pt}[\relax]\relax}


% draws a double rule by using a single rule row, a separator row, and then
% another single rule row 
% first optional argument determines the rule thicknesses, \arrayrulewidth is the default
% second optional argument determines the rule spacing, \doublerulesep is the default
% usage: \NCCPSeqnarraydblrulerow[rule line thickness][rule spacing]
\def\NCCPSeqnarraydblrulerow{\multispan{\@NCCPSeqnnumcols}\relax% span all the cols
% advance column counter only if the NCCPSeqnarray environment wants it
\if@advanceNCCPSeqncolcnt\global\advance\@NCCPSeqncolcnt by \@NCCPSeqnnumcols\relax\fi%
\@ifnextchar[{\@NCCPSeqnarraydblrulerow}{\@NCCPSeqnarraydblrulerow[\arrayrulewidth]}}
\def\@NCCPSeqnarraydblrulerow[#1]{\relax\@ifnextchar[{\@@NCCPSeqnarraydblrulerow[#1]}%
{\@@NCCPSeqnarraydblrulerow[#1][\doublerulesep]}}
\def\@@NCCPSeqnarraydblrulerow[#1][#2]{\def\@NCCPSeqnarraydblrulerowARG{#1}%
% we allow the user to say \NCCPSeqnarraydblrulerow[][]
\ifx\@NCCPSeqnarraydblrulerowARG\@empty%
\@NCCPSeqnarrayrulerow[\arrayrulewidth]%
\else%
\@NCCPSeqnarrayrulerow[#1]\relax%
\fi%
\def\@NCCPSeqnarraydblrulerowARG{#2}%
\ifx\@NCCPSeqnarraydblrulerowARG\@empty%
\\\NCCPSeqnarrayseprow[\doublerulesep][\relax]%
\else%
\\\NCCPSeqnarrayseprow[#2][\relax]%
\fi%
\\\multispan{\@NCCPSeqnnumcols}%
% advance column counter only if the NCCPSeqnarray environment wants it
\if@advanceNCCPSeqncolcnt\global\advance\@NCCPSeqncolcnt by \@NCCPSeqnnumcols\relax\fi%
\def\@NCCPSeqnarraydblrulerowARG{#1}%
\ifx\@NCCPSeqnarraydblrulerowARG\@empty%
\@NCCPSeqnarrayrulerow[\arrayrulewidth]%
\else%
\@NCCPSeqnarrayrulerow[#1]%
\fi%
}

% draws a double rule by using a single rule row, a separator (cutting) row, and then
% another single rule row 
% first optional argument determines the rule thicknesses, \arrayrulewidth is the default
% second optional argument determines the rule spacing, \doublerulesep is the default
% usage: \NCCPSeqnarraydblrulerow[rule line thickness][rule spacing]
\def\NCCPSeqnarraydblrulerowcut{\multispan{\@NCCPSeqnnumcols}\relax% span all the cols
% advance column counter only if the NCCPSeqnarray environment wants it
\if@advanceNCCPSeqncolcnt\global\advance\@NCCPSeqncolcnt by \@NCCPSeqnnumcols\relax\fi%
\@ifnextchar[{\@NCCPSeqnarraydblrulerowcut}{\@NCCPSeqnarraydblrulerowcut[\arrayrulewidth]}}
\def\@NCCPSeqnarraydblrulerowcut[#1]{\relax\@ifnextchar[{\@@NCCPSeqnarraydblrulerowcut[#1]}%
{\@@NCCPSeqnarraydblrulerowcut[#1][\doublerulesep]}}
\def\@@NCCPSeqnarraydblrulerowcut[#1][#2]{\def\@NCCPSeqnarraydblrulerowARG{#1}%
% we allow the user to say \NCCPSeqnarraydblrulerow[][]
\ifx\@NCCPSeqnarraydblrulerowARG\@empty%
\@NCCPSeqnarrayrulerow[\arrayrulewidth]%
\else%
\@NCCPSeqnarrayrulerow[#1]%
\fi%
\def\@NCCPSeqnarraydblrulerowARG{#2}%
\ifx\@NCCPSeqnarraydblrulerowARG\@empty%
\\\NCCPSeqnarrayseprowcut[\doublerulesep][\relax]%
\else%
\\\NCCPSeqnarrayseprowcut[#2][\relax]%
\fi%
\\\multispan{\@NCCPSeqnnumcols}%
% advance column counter only if the NCCPSeqnarray environment wants it
\if@advanceNCCPSeqncolcnt\global\advance\@NCCPSeqncolcnt by \@NCCPSeqnnumcols\relax\fi%
\def\@NCCPSeqnarraydblrulerowARG{#1}%
\ifx\@NCCPSeqnarraydblrulerowARG\@empty%
\@NCCPSeqnarrayrulerow[\arrayrulewidth]%
\else%
\@NCCPSeqnarrayrulerow[#1]%
\fi%
}



% inserts a full row's worth of &'s
% relies on \@NCCPSeqnnumcols to provide the correct number of columns
% uses \@NCCPStmptoksA, \count0 as scratch registers
\def\@NCCPSeqnarrayhoptolastcolumn{\@NCCPStmptoksA={}\count0=1\relax%
\loop% add cols if the user did not use them all
\ifnum\count0<\@NCCPSeqnnumcols\relax%
\@NCCPSappendtoksA{&}%
\advance\count0 by 1\relax% update the col count
\repeat%
\the\@NCCPStmptoksA%execute the &'s
}



\newif\if@NCCPSeqnarrayISinner % flag to indicate if we are within the lines
\@NCCPSeqnarrayISinnerfalse    % of an NCCPSeqnarray - after the NCCPSeqnarraydecl

\edef\@NCCPSeqnarrayTHEstrutheight{0pt} % height and depth of NCCPSeqnarray struts
\edef\@NCCPSeqnarrayTHEstrutdepth{0pt}

\edef\@NCCPSeqnarrayTHEmasterstrutheight{0pt} % default height and depth of
\edef\@NCCPSeqnarrayTHEmasterstrutdepth{0pt}  % struts within an NCCPSeqnarray

\edef\@NCCPSeqnarrayTHEmasterstrutHSAVE{0pt} % saved master strut height
\edef\@NCCPSeqnarrayTHEmasterstrutDSAVE{0pt} % and depth

\newif\if@NCCPSeqnarrayusemasterstrut % flag to indicate that the master strut value
\@NCCPSeqnarrayusemasterstruttrue     % is to be used



% saves the strut height and depth of the master strut
\def\@NCCPSeqnarraymasterstrutsave{\relax%
\expandafter\skip0=\@NCCPSeqnarrayTHEmasterstrutheight\relax%
\expandafter\skip2=\@NCCPSeqnarrayTHEmasterstrutdepth\relax%
% remove stretchability
\dimen0\skip0\relax%
\dimen2\skip2\relax%
% save values
\edef\@NCCPSeqnarrayTHEmasterstrutHSAVE{\the\dimen0}%
\edef\@NCCPSeqnarrayTHEmasterstrutDSAVE{\the\dimen2}}

% restores the strut height and depth of the master strut
\def\@NCCPSeqnarraymasterstrutrestore{\relax%
\expandafter\skip0=\@NCCPSeqnarrayTHEmasterstrutHSAVE\relax%
\expandafter\skip2=\@NCCPSeqnarrayTHEmasterstrutDSAVE\relax%
% remove stretchability
\dimen0\skip0\relax%
\dimen2\skip2\relax%
% restore values
\edef\@NCCPSeqnarrayTHEmasterstrutheight{\the\dimen0}%
\edef\@NCCPSeqnarrayTHEmasterstrutdepth{\the\dimen2}}


% globally restores the strut height and depth to the 
% master values and sets the master strut flag to true
\def\@NCCPSeqnarraystrutreset{\relax%
\expandafter\skip0=\@NCCPSeqnarrayTHEmasterstrutheight\relax%
\expandafter\skip2=\@NCCPSeqnarrayTHEmasterstrutdepth\relax%
% remove stretchability
\dimen0\skip0\relax%
\dimen2\skip2\relax%
% restore values
\xdef\@NCCPSeqnarrayTHEstrutheight{\the\dimen0}%
\xdef\@NCCPSeqnarrayTHEstrutdepth{\the\dimen2}%
\global\@NCCPSeqnarrayusemasterstruttrue}


% if the master strut is not to be used, make the current
% values of \@NCCPSeqnarrayTHEstrutheight, \@NCCPSeqnarrayTHEstrutdepth
% and the use master strut flag, global
% this allows user strut commands issued in the last column to be carried
% into the isolation/strut column
\def\@NCCPSeqnarrayglobalizestrutstatus{\relax%
\if@NCCPSeqnarrayusemasterstrut\else%
\xdef\@NCCPSeqnarrayTHEstrutheight{\@NCCPSeqnarrayTHEstrutheight}%
\xdef\@NCCPSeqnarrayTHEstrutdepth{\@NCCPSeqnarrayTHEstrutdepth}%
\global\@NCCPSeqnarrayusemasterstrutfalse%
\fi}



% usage: \NCCPSeqnarraystrutsize{height}{depth}[font size commands]
% If called outside the lines of an NCCPSeqnarray, sets the height
% and depth of both the master and local struts. If called inside
% an NCCPSeqnarray line, sets the height and depth of the local strut
% only and sets the flag to indicate the use of the local strut
% values. If the height or depth is left blank, 0.7\normalbaselineskip
% and 0.3\normalbaselineskip will be used, respectively.
% The optional argument can be used to evaluate the lengths under
% a different font size and styles. If none is specified, the current
% font is used.
% uses scratch registers \skip0, \skip2, \skip3, \dimen0, \dimen2
\def\NCCPSeqnarraystrutsize#1#2{\relax\@ifnextchar[{\@NCCPSeqnarraystrutsize{#1}{#2}}{\@NCCPSeqnarraystrutsize{#1}{#2}[\relax]}}
\def\@NCCPSeqnarraystrutsize#1#2[#3]{\def\@NCCPSeqnarraystrutsizeARG{#1}%
\ifx\@NCCPSeqnarraystrutsizeARG\@empty%
{\setbox0=\hbox{#3\relax\global\skip3=0.7\normalbaselineskip}}%
\skip0=\skip3\relax%
\else% arg one present
{\setbox0=\hbox{#3\relax\global\skip3=#1\relax}}%
\skip0=\skip3\relax%
\fi% if null arg
\def\@NCCPSeqnarraystrutsizeARG{#2}%
\ifx\@NCCPSeqnarraystrutsizeARG\@empty%
{\setbox0=\hbox{#3\relax\global\skip3=0.3\normalbaselineskip}}%
\skip2=\skip3\relax%
\else% arg two present
{\setbox0=\hbox{#3\relax\global\skip3=#2\relax}}%
\skip2=\skip3\relax%
\fi% if null arg
% remove stretchability, just to be safe
\dimen0\skip0\relax%
\dimen2\skip2\relax%
% dimen0 = height, dimen2 = depth
\if@NCCPSeqnarrayISinner% inner does not touch master strut size
\edef\@NCCPSeqnarrayTHEstrutheight{\the\dimen0}%
\edef\@NCCPSeqnarrayTHEstrutdepth{\the\dimen2}%
\@NCCPSeqnarrayusemasterstrutfalse% do not use master
\else% outer, have to set master strut too
\edef\@NCCPSeqnarrayTHEmasterstrutheight{\the\dimen0}%
\edef\@NCCPSeqnarrayTHEmasterstrutdepth{\the\dimen2}%
\edef\@NCCPSeqnarrayTHEstrutheight{\the\dimen0}%
\edef\@NCCPSeqnarrayTHEstrutdepth{\the\dimen2}%
\@NCCPSeqnarrayusemasterstruttrue% use master strut
\fi}


% usage: \NCCPSeqnarraystrutsizeadd{added height}{added depth}[font size commands]
% If called outside the lines of an NCCPSeqnarray, adds the given height
% and depth to both the master and local struts.
% If called inside an NCCPSeqnarray line, adds the given height and depth
% to the local strut only and sets the flag to indicate the use 
% of the local strut values.
% In both cases, if a height or depth is left blank, 0pt is used instead.
% The optional argument can be used to evaluate the lengths under
% a different font size and styles. If none is specified, the current
% font is used.
% uses scratch registers \skip0, \skip2, \skip3, \dimen0, \dimen2
\def\NCCPSeqnarraystrutsizeadd#1#2{\relax\@ifnextchar[{\@NCCPSeqnarraystrutsizeadd{#1}{#2}}{\@NCCPSeqnarraystrutsizeadd{#1}{#2}[\relax]}}
\def\@NCCPSeqnarraystrutsizeadd#1#2[#3]{\def\@NCCPSeqnarraystrutsizearg{#1}%
\ifx\@NCCPSeqnarraystrutsizearg\@empty%
\skip0=0pt\relax%
\else% arg one present
{\setbox0=\hbox{#3\relax\global\skip3=#1}}%
\skip0=\skip3\relax%
\fi% if null arg
\def\@NCCPSeqnarraystrutsizearg{#2}%
\ifx\@NCCPSeqnarraystrutsizearg\@empty%
\skip2=0pt\relax%
\else% arg two present
{\setbox0=\hbox{#3\relax\global\skip3=#2}}%
\skip2=\skip3\relax%
\fi% if null arg
% remove stretchability, just to be safe
\dimen0\skip0\relax%
\dimen2\skip2\relax%
% dimen0 = height, dimen2 = depth
\if@NCCPSeqnarrayISinner% inner does not touch master strut size
% get local strut size
\expandafter\skip0=\@NCCPSeqnarrayTHEstrutheight\relax%
\expandafter\skip2=\@NCCPSeqnarrayTHEstrutdepth\relax%
% add it to the user supplied values
\advance\dimen0 by \skip0\relax%
\advance\dimen2 by \skip2\relax%
% update the local strut size
\edef\@NCCPSeqnarrayTHEstrutheight{\the\dimen0}%
\edef\@NCCPSeqnarrayTHEstrutdepth{\the\dimen2}%
\@NCCPSeqnarrayusemasterstrutfalse% do not use master
\else% outer, have to set master strut too
% get master strut size
\expandafter\skip0=\@NCCPSeqnarrayTHEmasterstrutheight\relax%
\expandafter\skip2=\@NCCPSeqnarrayTHEmasterstrutdepth\relax%
% add it to the user supplied values
\advance\dimen0 by \skip0\relax%
\advance\dimen2 by \skip2\relax%
% update the local and master strut sizes
\edef\@NCCPSeqnarrayTHEmasterstrutheight{\the\dimen0}%
\edef\@NCCPSeqnarrayTHEmasterstrutdepth{\the\dimen2}%
\edef\@NCCPSeqnarrayTHEstrutheight{\the\dimen0}%
\edef\@NCCPSeqnarrayTHEstrutdepth{\the\dimen2}%
\@NCCPSeqnarrayusemasterstruttrue% use master strut
\fi}


% allow user a way to see the struts
\newif\ifNCCPSvisiblestruts
\NCCPSvisiblestrutsfalse

% inserts an invisible strut using the master or local strut values
% uses scratch registers \skip0, \skip2, \dimen0, \dimen2
\def\@NCCPSeqnarrayinsertstrut{\relax%
\if@NCCPSeqnarrayusemasterstrut
% get master strut size
\expandafter\skip0=\@NCCPSeqnarrayTHEmasterstrutheight\relax%
\expandafter\skip2=\@NCCPSeqnarrayTHEmasterstrutdepth\relax%
\else%
% get local strut size
\expandafter\skip0=\@NCCPSeqnarrayTHEstrutheight\relax%
\expandafter\skip2=\@NCCPSeqnarrayTHEstrutdepth\relax%
\fi%
% remove stretchability, probably not needed
\dimen0\skip0\relax%
\dimen2\skip2\relax%
% dimen0 = height, dimen2 = depth
% allow user to see struts if desired
\ifNCCPSvisiblestruts%
\vrule width0.2pt height\dimen0 depth\dimen2\relax%
\else%
\vrule width0pt height\dimen0 depth\dimen2\relax\fi}


% creates an invisible strut, useable even outside \NCCPSeqnarray
% if \NCCPSvisiblestrutstrue, the strut will be visible and 0.2pt wide. 
% usage: \NCCPSstrut[height][depth][font size commands]
% default is \NCCPSstrut[0.7\normalbaselineskip][0.3\normalbaselineskip][\relax]
% blank arguments inherit the default values
% uses \dimen0, \dimen2, \skip0, \skip2
\def\NCCPSstrut{\relax\@ifnextchar[{\@NCCPSstrut}{\@NCCPSstrut[0.7\normalbaselineskip]}}
\def\@NCCPSstrut[#1]{\relax\@ifnextchar[{\@@NCCPSstrut[#1]}{\@@NCCPSstrut[#1][0.3\normalbaselineskip]}}
\def\@@NCCPSstrut[#1][#2]{\relax\@ifnextchar[{\@@@NCCPSstrut[#1][#2]}{\@@@NCCPSstrut[#1][#2][\relax]}}
\def\@@@NCCPSstrut[#1][#2][#3]{\mbox{#3\relax%
\def\@NCCPSstrutARG{#1}%
\ifx\@NCCPSstrutARG\@empty%
\skip0=0.7\normalbaselineskip\relax%
\else%
\skip0=#1\relax%
\fi%
\def\@NCCPSstrutARG{#2}%
\ifx\@NCCPSstrutARG\@empty%
\skip2=0.3\normalbaselineskip\relax%
\else%
\skip2=#2\relax%
\fi%
% remove stretchability, probably not needed
\dimen0\skip0\relax%
\dimen2\skip2\relax%
\ifNCCPSvisiblestruts%
\vrule width0.2pt height\dimen0 depth\dimen2\relax%
\else%
\vrule width0.0pt height\dimen0 depth\dimen2\relax\fi}}


% enables strut mode by setting a default strut size and then zeroing the
% \baselineskip, \lineskip, \lineskiplimit and \jot
\def\NCCPSeqnarraystrutmode{\NCCPSeqnarraystrutsize{0.7\normalbaselineskip}{0.3\normalbaselineskip}[\relax]%
\baselineskip=0pt\lineskip=0pt\lineskiplimit=0pt\jot=0pt}


% equation and subequation forms to use to setup hyperref's \@currentHref
\def\@NCCPStheHrefequation{equation.\theHequation}
\def\@NCCPStheHrefsubequation{equation.\theHequation\alph{NCCPSsubequation}}


\def\NCCPSeqnarray{\@NCCPSeqnumpersisttrue\@NCCPSsubeqnumpersistfalse\@NCCPSeqnarray}
\def\endNCCPSeqnarray{\end@NCCPSeqnarray}

\@namedef{NCCPSeqnarray*}{\@NCCPSeqnumpersistfalse\@NCCPSsubeqnumpersistfalse\@NCCPSeqnarray}
\@namedef{endNCCPSeqnarray*}{\end@NCCPSeqnarray}


% \NCCPSeqnarray is an enhanced \eqnarray. 
% The star form defaults to not putting equation numbers at the end of each row.
% usage: \NCCPSeqnarray[decl]{cols}
\def\@NCCPSeqnarray{\relax\@ifnextchar[{\@@NCCPSeqnarray}{\@@NCCPSeqnarray[\relax]}}
% We have to be careful here to normalize catcodes just before acquiring the
% cols as that specification may contain punctuation which could be subject
% to document catcode changes.
\def\@@NCCPSeqnarray[#1]{\begingroup\NCCPSnormalcatcodes\@@@NCCPSeqnarray[#1]}
\def\@@@NCCPSeqnarray[#1]#2{\endgroup
   % default to showing the equation number or not based on whether or not
   % the star form was involked
   \if@NCCPSeqnumpersist\global\@eqnswtrue
   \else% not the star form
   \global\@eqnswfalse
   \fi% if star form
   % provide a basic hyperref \theHequation if this has not already been setup (hyperref not loaded, or no section counter)
   \@ifundefined{theHequation}{\def\theHequation{\arabic{equation}}}{}\relax
   % provide dummy hyperref commands in case hyperref is not loaded
   \providecommand{\Hy@raisedlink}[1]{}\relax
   \providecommand{\hyper@anchorstart}[1]{}\relax
   \providecommand{\hyper@anchorend}{}\relax
   \providecommand{\@currentHref}{}\relax
   \@NCCPSeqnumpreadvfalse% reset eqnpreadv flag
   \@NCCPSsubeqnumpreadvfalse% reset subeqnpreadv flag
   \@NCCPSeqnarrayISinnerfalse% not yet within the lines of the halign
   \@NCCPSeqnarraystrutsize{0pt}{0pt}[\relax]% turn off struts by default
   \@NCCPSeqnarrayusemasterstruttrue% use master strut till user asks otherwise
   \NCCPSvisiblestrutsfalse% diagnostic mode defaults to off
   % no extra space unless the user specifically requests it
   \lineskip=0pt\relax
   \lineskiplimit=0pt\relax
   \baselineskip=\normalbaselineskip\relax%
   \jot=\NCCPSnormaljot\relax%
   \mathsurround\z@\relax% no extra spacing around math
   \@advanceNCCPSeqncolcnttrue% advance the col counter for each col the user uses, 
                             % used in \NCCPSeqnarraymulticol and in the preamble build
   %V1.8 Here we preadvance to the next equation number.
   % If the user later wants a continued subequation, we can roll back.
   \global\@NCCPSsubeqnnumrollback=\c@NCCPSsubequation%
   \stepcounter{equation}\@NCCPSeqnumpreadvtrue% advance equation counter before first line
   \setcounter{NCCPSsubequation}{0}% no subequation yet
   \let\@NCCPScurrentlabelsave\@currentlabel% save current label as we later change it globally
   \let\@NCCPScurrentHrefsave\@currentHref% save current href label as we later change it globally
   \def\@currentlabel{\p@equation\theequation}% redefine the ref label
   \def\@currentHref{\@NCCPStheHrefequation}% setup hyperref label
   \NCCPSeqnarraydecl\relax% allow a way for the user to make global overrides
   #1\relax% allow user to override defaults
   \let\\\@NCCPSeqnarraycr% replace newline with one that can put in eqn. numbers
   \global\@NCCPSeqncolcnt\z@% col. count = 0 for first line
   \@NCCPSbuildpreamble{#2}\relax% build the preamble and put it into \@NCCPStmptoksA 
   % put in the column for the equation number
   \ifnum\@NCCPSeqnnumcols>0\relax\@NCCPSappendtoksA{&}\fi% col separator for those after the first
   \toks0={##}%
   % advance the \@NCCPSeqncolcnt for the isolation col, this helps with error checking
   \@NCCPSappendtoksA{\global\advance\@NCCPSeqncolcnt by 1\relax}%
   % add the isolation column
   \@NCCPSappendtoksA{\tabskip\z@skip\bgroup\the\toks0\egroup}%
   % advance the \@NCCPSeqncolcnt for the equation number col, this helps with error checking
   \@NCCPSappendtoksA{&\global\advance\@NCCPSeqncolcnt by 1\relax}%
   % add the equation number col to the preamble
   \@NCCPSappendtoksA{\tabskip\z@skip\hb@xt@\z@\bgroup\hss\the\toks0\egroup}%
   % note \@NCCPSeqnnumcols does not count the equation col or isolation col
   % set the starting tabskip glue as determined by the preamble build
   \tabskip=\@NCCPSBPstartglue\relax
   % begin the display alignment
   \@NCCPSeqnarrayISinnertrue% commands are now within the lines
   $$\everycr{}\halign to\displaywidth\bgroup
   % "exspand" the preamble
   \span\the\@NCCPStmptoksA\cr}

% enter isolation/strut column (or the next column if the user did not use
% every column), record the strut status, complete the columns, do the strut if needed,
% restore counters (to backout any equation setup for a next line that was never used)
% to their correct values and exit
\def\end@NCCPSeqnarray{\@NCCPSeqnarrayglobalizestrutstatus&\@@NCCPSeqnarraycr\egroup
\if@NCCPSsubeqnumpreadv\global\advance\c@NCCPSsubequation\m@ne\fi
\if@NCCPSeqnumpreadv\global\advance\c@equation\m@ne\global\c@NCCPSsubequation=\@NCCPSsubeqnnumrollback\fi
\global\let\@currentlabel\@NCCPScurrentlabelsave% restore current label
\global\let\@currentHref\@NCCPScurrentHrefsave% restore current href label
$$\@ignoretrue}


% NCCPSeqnarray uses a modifed \\ instead of the plain \cr to
% end rows. This allows for things like \\*[vskip amount]
% These "cr" macros are modified versions of those for LaTeX2e's eqnarray
% the {\ifnum0=`} braces must be kept away from the last column to avoid
% altering spacing of its math, so we use & to advance to the next column
% as there is an isolation/strut column after the user's columns
\def\@NCCPSeqnarraycr{\@NCCPSeqnarrayglobalizestrutstatus&% save strut status and advance to next column
   {\ifnum0=`}\fi
   \@ifstar{%
      \global\@eqpen\@M\@NCCPSeqnarrayYCR
   }{%
      \global\@eqpen\interdisplaylinepenalty \@NCCPSeqnarrayYCR
   }%
}

\def\@NCCPSeqnarrayYCR{\@testopt\@NCCPSeqnarrayXCR\z@skip}

\def\@NCCPSeqnarrayXCR[#1]{%
   \ifnum0=`{\fi}%
   \@@NCCPSeqnarraycr
   \noalign{\penalty\@eqpen\vskip\jot\vskip #1\relax}}%

\def\@@NCCPSeqnarraycr{\@NCCPStmptoksA={}% clear token register
    \advance\@NCCPSeqncolcnt by -1\relax% adjust col count because of the isolation column
    \ifnum\@NCCPSeqncolcnt>\@NCCPSeqnnumcols\relax
    \@NCCPSclspkgerror{Too many columns within the NCCPSeqnarray\MessageBreak
                          environment}%
    {Use fewer \string &'s or put more columns in the NCCPSeqnarray column\MessageBreak 
     specifications.}\relax%
    \else
    \loop% add cols if the user did not use them all
    \ifnum\@NCCPSeqncolcnt<\@NCCPSeqnnumcols\relax
    \@NCCPSappendtoksA{&}%
    \advance\@NCCPSeqncolcnt by 1\relax% update the col count
    \repeat
    % this number of &'s will take us the the isolation column
    \fi
    % execute the &'s
    \the\@NCCPStmptoksA%
    % handle the strut/isolation column
    \@NCCPSeqnarrayinsertstrut% do the strut if needed
    \@NCCPSeqnarraystrutreset% reset the strut system for next line or NCCPSeqnarray
    &% and enter the equation number column
    \if@eqnsw% only if we display something
      \Hy@raisedlink{\hyper@anchorstart{\@currentHref}}% start a hyperref anchor
      \global\@NCCPSeqnumpreadvfalse\relax% displaying an equation number means
      \global\@NCCPSsubeqnumpreadvfalse\relax% the equation counters point to valid equations
      % V1.8 Here we setup the counters, currentlabel and status for what would be the *next*
      % equation line as would be the case under the current settings. However, there are two problems.
      % One problem is that there might not ever be a next line. The second problem is that the user
      % may later alter the meaning of a line with commands such as \NCCPSyessubnumber. So, to handle
      % these cases we have to record the current values of the (sub)equation counters and revert back
      % to them if the next line is changed or never comes. The \if@NCCPSeqnumpreadv, \if@NCCPSsubeqnumpreadv
      % and \@NCCPSsubeqnnumrollback stuff tracks this.
      % The logic to handle all this is surprisingly complex, but a nice feature of the approach here is
      % that the equation counters and labels remain valid for what the line would be unless a
      % \NCCPSyessubnumber et al. later changes it. So, any hyperref links are always correct.
      \ifnum\c@NCCPSsubequation>0\relax% handle subequation
         \theNCCPSsubequationdis\relax
         \if@NCCPSsubeqnumpersist% setup for default type of next line
            \stepcounter{NCCPSsubequation}\global\@NCCPSsubeqnumpreadvtrue\relax
            \gdef\@currentlabel{\p@NCCPSsubequation\theNCCPSsubequation}\relax
            \gdef\@currentHref{\@NCCPStheHrefsubequation}% setup hyperref label
         \else
             % if no subeqnum persist, go ahead and setup for a new equation number
             \global\@NCCPSsubeqnnumrollback=\c@NCCPSsubequation
             \stepcounter{equation}\global\@NCCPSeqnumpreadvtrue\relax
             \setcounter{NCCPSsubequation}{0}\gdef\@currentlabel{\p@equation\theequation}\relax
             \gdef\@currentHref{\@NCCPStheHrefequation}% setup hyperref label
         \fi
      \else% display a standard equation number
        \theequationdis\relax
        \setcounter{NCCPSsubequation}{0}\relax% not really needed
        \if@NCCPSsubeqnumpersist% setup for default type of next line
           % subequations that follow plain equations carry the same equation number e.g, 5, 5a rather than 5, 6a
           \stepcounter{NCCPSsubequation}\global\@NCCPSsubeqnumpreadvtrue\relax
           \gdef\@currentlabel{\p@NCCPSsubequation\theNCCPSsubequation}\relax
           \gdef\@currentHref{\@NCCPStheHrefsubequation}% setup hyperref label
         \else
             % if no subeqnum persist, go ahead and setup for a new equation number
             \global\@NCCPSsubeqnnumrollback=\c@NCCPSsubequation
             \stepcounter{equation}\global\@NCCPSeqnumpreadvtrue\relax
             \setcounter{NCCPSsubequation}{0}\gdef\@currentlabel{\p@equation\theequation}\relax
             \gdef\@currentHref{\@NCCPStheHrefequation}% setup hyperref label
         \fi
      \fi%
      \Hy@raisedlink{\hyper@anchorend}% end hyperref anchor
    \fi% fi only if we display something
    % reset the flags to indicate the default preferences of the display of equation numbers
    \if@NCCPSeqnumpersist\global\@eqnswtrue\else\global\@eqnswfalse\fi
    \if@NCCPSsubeqnumpersist\global\@eqnswtrue\fi% ditto for the subequation flag
    % reset the number of columns the user actually used
    \global\@NCCPSeqncolcnt\z@\relax
    % the real end of the line
    \cr}





% \NCCPSeqnarraybox is like \NCCPSeqnarray except the box form puts everything
% inside a vtop, vbox, or vcenter box depending on the letter in the second
% optional argument (t,b,c). Vbox is the default. Unlike \NCCPSeqnarray,
% equation numbers are not displayed and \NCCPSeqnarraybox can be nested.
% \NCCPSeqnarrayboxm is for math mode (like \array) and does not put the vbox
% within an hbox.
% \NCCPSeqnarrayboxt is for text mode (like \tabular) and puts the vbox within
% a \hbox{$ $} construct.
% \NCCPSeqnarraybox will auto detect whether to use \NCCPSeqnarrayboxm or 
% \NCCPSeqnarrayboxt depending on the math mode.
% The third optional argument specifies the width this box is to be set to -
% natural width is the default.
% The * forms do not add \jot line spacing
% usage: \NCCPSeqnarraybox[decl][pos][width]{cols}
\def\NCCPSeqnarrayboxm{\@NCCPSeqnarrayboxnojotfalse\@NCCPSeqnarrayboxHBOXSWfalse\@NCCPSeqnarraybox}
\def\endNCCPSeqnarrayboxm{\end@NCCPSeqnarraybox}
\@namedef{NCCPSeqnarrayboxm*}{\@NCCPSeqnarrayboxnojottrue\@NCCPSeqnarrayboxHBOXSWfalse\@NCCPSeqnarraybox}
\@namedef{endNCCPSeqnarrayboxm*}{\end@NCCPSeqnarraybox}

\def\NCCPSeqnarrayboxt{\@NCCPSeqnarrayboxnojotfalse\@NCCPSeqnarrayboxHBOXSWtrue\@NCCPSeqnarraybox}
\def\endNCCPSeqnarrayboxt{\end@NCCPSeqnarraybox}
\@namedef{NCCPSeqnarrayboxt*}{\@NCCPSeqnarrayboxnojottrue\@NCCPSeqnarrayboxHBOXSWtrue\@NCCPSeqnarraybox}
\@namedef{endNCCPSeqnarrayboxt*}{\end@NCCPSeqnarraybox}

\def\NCCPSeqnarraybox{\@NCCPSeqnarrayboxnojotfalse\ifmmode\@NCCPSeqnarrayboxHBOXSWfalse\else\@NCCPSeqnarrayboxHBOXSWtrue\fi%
\@NCCPSeqnarraybox}
\def\endNCCPSeqnarraybox{\end@NCCPSeqnarraybox}

\@namedef{NCCPSeqnarraybox*}{\@NCCPSeqnarrayboxnojottrue\ifmmode\@NCCPSeqnarrayboxHBOXSWfalse\else\@NCCPSeqnarrayboxHBOXSWtrue\fi%
\@NCCPSeqnarraybox}
\@namedef{endNCCPSeqnarraybox*}{\end@NCCPSeqnarraybox}

% flag to indicate if the \NCCPSeqnarraybox needs to put things into an hbox{$ $} 
% for \vcenter in non-math mode
\newif\if@NCCPSeqnarrayboxHBOXSW%
\@NCCPSeqnarrayboxHBOXSWfalse

\def\@NCCPSeqnarraybox{\relax\@ifnextchar[{\@@NCCPSeqnarraybox}{\@@NCCPSeqnarraybox[\relax]}}
% We have to be careful here to normalize catcodes just before acquiring the
% cols as that specification may contain punctuation which could be subject
% to document catcode changes.
\def\@@NCCPSeqnarraybox[#1]{\relax\begingroup\NCCPSnormalcatcodes\@ifnextchar[{\@@@NCCPSeqnarraybox[#1]}{\@@@NCCPSeqnarraybox[#1][b]}}
\def\@@@NCCPSeqnarraybox[#1][#2]{\relax\@ifnextchar[{\@@@@NCCPSeqnarraybox[#1][#2]}{\@@@@NCCPSeqnarraybox[#1][#2][\relax]}}

% #1 = decl; #2 = t,b,c; #3 = width, #4 = col specs
\def\@@@@NCCPSeqnarraybox[#1][#2][#3]#4{\endgroup\@NCCPSeqnarrayISinnerfalse % not yet within the lines of the halign
   \@NCCPSeqnarraymasterstrutsave% save current master strut values
   \@NCCPSeqnarraystrutsize{0pt}{0pt}[\relax]% turn off struts by default
   \@NCCPSeqnarrayusemasterstruttrue% use master strut till user asks otherwise
   \NCCPSvisiblestrutsfalse% diagnostic mode defaults to off
   % no extra space unless the user specifically requests it
   \lineskip=0pt\relax%
   \lineskiplimit=0pt\relax%
   \baselineskip=\normalbaselineskip\relax%
   \jot=\NCCPSnormaljot\relax%
   \mathsurround\z@\relax% no extra spacing around math
   % the default end glues are zero for an \NCCPSeqnarraybox
   \edef\@NCCPSeqnarraycolSEPdefaultstart{\@NCCPSeqnarraycolSEPzero}% default start glue
   \edef\@NCCPSeqnarraycolSEPdefaultend{\@NCCPSeqnarraycolSEPzero}% default end glue
   \edef\@NCCPSeqnarraycolSEPdefaultmid{\@NCCPSeqnarraycolSEPzero}% default inter-column glue
   \@advanceNCCPSeqncolcntfalse% do not advance the col counter for each col the user uses, 
                              % used in \NCCPSeqnarraymulticol and in the preamble build
   \NCCPSeqnarrayboxdecl\relax% allow a way for the user to make global overrides
   #1\relax% allow user to override defaults
   \let\\\@NCCPSeqnarrayboxcr% replace newline with one that allows optional spacing
   \@NCCPSbuildpreamble{#4}\relax% build the preamble and put it into \@NCCPStmptoksA
   % add an isolation column to the preamble to stop \\'s {} from getting into the last col
   \ifnum\@NCCPSeqnnumcols>0\relax\@NCCPSappendtoksA{&}\fi% col separator for those after the first
   \toks0={##}%
   % add the isolation column to the preamble
   \@NCCPSappendtoksA{\tabskip\z@skip\bgroup\the\toks0\egroup}% 
   % set the starting tabskip glue as determined by the preamble build
   \tabskip=\@NCCPSBPstartglue\relax
   % begin the alignment
   \everycr{}%
   % use only the very first token to determine the positioning
   \@NCCPSextracttoken{#2}\relax
   \ifx\@NCCPSextractedtokensdiscarded\@empty\else
     \typeout{** WARNING: NCCPSeqnarraybox position specifiers after the first in `\@NCCPSextracttokenarg' ignored (line \the\inputlineno).}\relax
   \fi
   % \@NCCPSextractedtoken has the first token, the rest are ignored
   % if we need to put things into and hbox and go into math mode, do so now
   \if@NCCPSeqnarrayboxHBOXSW \leavevmode \hbox \bgroup $\fi%
   % use the appropriate vbox type
   \if\@NCCPSextractedtoken t\relax\vtop\else\if\@NCCPSextractedtoken c\relax%
   \vcenter\else\vbox\fi\fi\bgroup%
   \@NCCPSeqnarrayISinnertrue% commands are now within the lines
   \ifx#3\relax\halign\else\halign to #3\relax\fi%
   \bgroup
   % "exspand" the preamble
   \span\the\@NCCPStmptoksA\cr}

% carry strut status and enter the isolation/strut column, 
% exit from math mode if needed, and exit
\def\end@NCCPSeqnarraybox{\@NCCPSeqnarrayglobalizestrutstatus% carry strut status
&% enter isolation/strut column
\@NCCPSeqnarrayinsertstrut% do strut if needed
\@NCCPSeqnarraymasterstrutrestore% restore the previous master strut values
% reset the strut system for next NCCPSeqnarray
% (sets local strut values back to previous master strut values)
\@NCCPSeqnarraystrutreset%
% ensure last line, exit from halign, close vbox
\crcr\egroup\egroup%
% exit from math mode and close hbox if needed
\if@NCCPSeqnarrayboxHBOXSW $\egroup\fi}



% NCCPSeqnarraybox uses a modifed \\ instead of the plain \cr to
% end rows. This allows for things like \\[vskip amount]
% This "cr" macros are modified versions those for LaTeX2e's eqnarray
% For NCCPSeqnarraybox, \\* is the same as \\
% the {\ifnum0=`} braces must be kept away from the last column to avoid
% altering spacing of its math, so we use & to advance to the isolation/strut column
% carry strut status into isolation/strut column
\def\@NCCPSeqnarrayboxcr{\@NCCPSeqnarrayglobalizestrutstatus% carry strut status
&% enter isolation/strut column
\@NCCPSeqnarrayinsertstrut% do strut if needed
% reset the strut system for next line or NCCPSeqnarray
\@NCCPSeqnarraystrutreset%
{\ifnum0=`}\fi%
\@ifstar{\@NCCPSeqnarrayboxYCR}{\@NCCPSeqnarrayboxYCR}}

% test and setup the optional argument to \\[]
\def\@NCCPSeqnarrayboxYCR{\@testopt\@NCCPSeqnarrayboxXCR\z@skip}

% NCCPSeqnarraybox does not automatically increase line spacing by \jot
\def\@NCCPSeqnarrayboxXCR[#1]{\ifnum0=`{\fi}%
\cr\noalign{\if@NCCPSeqnarrayboxnojot\else\vskip\jot\fi\vskip#1\relax}}



% usage: \@NCCPSbuildpreamble{column specifiers}
% starts the halign preamble build 
% the assembled preamble is put in \@NCCPStmptoksA
\def\@NCCPSbuildpreamble#1{\@NCCPStmptoksA={}% clear token register
\let\@NCCPSBPcurtype=u%current column type is not yet known
\let\@NCCPSBPprevtype=s%the previous column type was the start
\let\@NCCPSBPnexttype=u%next column type is not yet known
% ensure these are valid
\def\@NCCPSBPcurglue={0pt plus 0pt minus 0pt}%
\def\@NCCPSBPcurcolname{@NCCPSdefault}% name of current column definition
% currently acquired numerically referenced glue
% use a name that is easier to remember
\let\@NCCPSBPcurnum=\@NCCPStmpcountA%
\@NCCPSBPcurnum=0%
% tracks number of columns in the preamble
\@NCCPSeqnnumcols=0%
% record the default end glues
\edef\@NCCPSBPstartglue{\@NCCPSeqnarraycolSEPdefaultstart}%
\edef\@NCCPSBPendglue{\@NCCPSeqnarraycolSEPdefaultend}%
\edef\@NCCPSedefMACRO{#1}\relax% fully expand the preamble to support macro containers
% now parse the user's column specifications
% \ignorespaces is used as a delimiter, need at least one trailing \relax because
% \@@NCCPSbuildpreamble looks into the future 
\expandafter\@@NCCPSbuildpreamble\@NCCPSedefMACRO\ignorespaces\relax\relax}


% usage: \@@NCCPSbuildpreamble{current column}{next column}
% parses and builds the halign preamble
\def\@@NCCPSbuildpreamble#1#2{\let\@@nextNCCPSbuildpreamble=\@@NCCPSbuildpreamble%
% use only the very first token to check the end
\@NCCPSextracttokengroups{#1}\relax
\ifx\@NCCPSextractedfirsttoken\ignorespaces\let\@@nextNCCPSbuildpreamble=\@@NCCPSfinishpreamble\else%
% identify current and next token type
\@NCCPSgetcoltype{#1}{\@NCCPSBPcurtype}{1}% current, error on invalid
\@NCCPSgetcoltype{#2}{\@NCCPSBPnexttype}{0}% next, no error on invalid next
% if curtype is a glue, get the glue def
\if\@NCCPSBPcurtype g\@NCCPSgetcurglue{#1}{\@NCCPSBPcurglue}\fi%
% if curtype is a column, get the column def and set the current column name
\if\@NCCPSBPcurtype c\@NCCPSgetcurcol{#1}\fi%
% if curtype is a numeral, acquire the user defined glue
\if\@NCCPSBPcurtype n\@NCCPSprocessNcol{#1}\fi%
% process the acquired glue 
\if\@NCCPSBPcurtype g\@NCCPSprocessGcol\fi%
% process the acquired col 
\if\@NCCPSBPcurtype c\@NCCPSprocessCcol\fi%
% ready prevtype for next col spec.
\let\@NCCPSBPprevtype=\@NCCPSBPcurtype%
% be sure and put back the future token(s) as a group
\fi\@@nextNCCPSbuildpreamble{#2}}


% usage: \@@NCCPSfinishpreamble{discarded}
% executed just after preamble build is completed
% warn about zero cols, and if prevtype type = u, put in end tabskip glue
% argument is not used
\def\@@NCCPSfinishpreamble#1{\ifnum\@NCCPSeqnnumcols<1\relax
\@NCCPSclspkgerror{No column specifiers declared for NCCPSeqnarray}%
{At least one column type must be declared for each NCCPSeqnarray.}%
\fi%num cols less than 1
%if last type undefined, set default end tabskip glue
\if\@NCCPSBPprevtype u\@NCCPSappendtoksA{\tabskip=\@NCCPSBPendglue}\fi}


% usage: \@NCCPSgetcoltype{col specifier}{\output}{error more}
% Identify and return the column specifier's type code in the given
% \output macro:
% n = number
% g = glue (any other char in catagory 12)
% c = letter
% e = \ignorespaces (end of sequence)
% u = undefined
% error mode: 0 = no error message, 1 = error on invalid char
\def\@NCCPSgetcoltype#1#2#3{%
% use only the very first token to determine the type
\@NCCPSextracttoken{#1}\relax
% \@NCCPSextractedtoken has the first token, the rest are discarded
\let#2=u\relax% assume invalid until know otherwise
\ifx\@NCCPSextractedtoken\ignorespaces\let#2=e\else
\ifcat\@NCCPSextractedtoken\relax\else% screen out control sequences
\if0\@NCCPSextractedtoken\let#2=n\else
\if1\@NCCPSextractedtoken\let#2=n\else
\if2\@NCCPSextractedtoken\let#2=n\else
\if3\@NCCPSextractedtoken\let#2=n\else
\if4\@NCCPSextractedtoken\let#2=n\else
\if5\@NCCPSextractedtoken\let#2=n\else
\if6\@NCCPSextractedtoken\let#2=n\else
\if7\@NCCPSextractedtoken\let#2=n\else
\if8\@NCCPSextractedtoken\let#2=n\else
\if9\@NCCPSextractedtoken\let#2=n\else
\ifcat,\@NCCPSextractedtoken\let#2=g\relax
\else\ifcat a\@NCCPSextractedtoken\let#2=c\relax\fi\fi\fi\fi\fi\fi\fi\fi\fi\fi\fi\fi\fi\fi
\if#2u\relax
\if0\noexpand#3\relax\else\@NCCPSclspkgerror{Invalid character in column specifications}%
{Only letters, numerals and certain other symbols are allowed \MessageBreak
as NCCPSeqnarray column specifiers.}\fi\fi}


% usage: \@NCCPSgetcurcol{col specifier}
% verify the letter referenced column exists
% and return its name in \@NCCPSBPcurcolname
% if column specifier is invalid, use the default column @NCCPSdefault
\def\@NCCPSgetcurcol#1{\expandafter\ifx\csname @NCCPSeqnarraycolDEF#1\endcsname\@NCCPSeqnarraycolisdefined%
\def\@NCCPSBPcurcolname{#1}\else% invalid column name
\@NCCPSclspkgerror{Invalid column type "#1" in column specifications.\MessageBreak
Using a default centering column instead}%
{You must define NCCPSeqnarray column types before use.}%
\def\@NCCPSBPcurcolname{@NCCPSdefault}\fi}


% usage: \@NCCPSgetcurglue{glue specifier}{\output}
% identify the predefined (punctuation) glue value
% and return it in the given output macro
\def\@NCCPSgetcurglue#1#2{%
% ! = \! (neg small)  -0.16667em (-3/18 em)
% , = \, (small)       0.16667em ( 3/18 em)
% : = \: (med)         0.22222em ( 4/18 em)
% ; = \; (large)       0.27778em ( 5/18 em)
% ' = \quad            1em
% " = \qquad           2em
% . = 0.5\arraycolsep
% / = \arraycolsep
% ? = 2\arraycolsep
% * = 1fil
% + = \@NCCPSeqnarraycolSEPcenter
% - = \@NCCPSeqnarraycolSEPzero
% Note that all em values are referenced to the math font (textfont2) fontdimen6
% value for 1em.
% 
% use only the very first token to determine the type
\@NCCPSextracttoken{#1}\relax
\ifx\@NCCPSextractedtokensdiscarded\@empty\else
  \typeout{** WARNING: NCCPSeqnarray predefined inter-column glue type specifiers after the first in `\@NCCPSextracttokenarg' ignored (line \the\inputlineno).}\relax
\fi
% get the math font 1em value
% LaTeX2e's NFSS2 does not preload the fonts, but \NCCPSeqnarray needs
% to gain access to the math (\textfont2) font's spacing parameters.
% So we create a bogus box here that uses the math font to ensure
% that \textfont2 is loaded and ready. If this is not done,
% the \textfont2 stuff here may not work.
% Thanks to Bernd Raichle for his 1997 post on this topic.
{\setbox0=\hbox{$\displaystyle\relax$}}%
% fontdimen6 has the width of 1em (a quad).
\@NCCPStmpdimenA=\fontdimen6\textfont2\relax%
% identify the glue value based on the first token
% we discard anything after the first
\if!\@NCCPSextractedtoken\@NCCPStmpdimenA=-0.16667\@NCCPStmpdimenA\edef#2{\the\@NCCPStmpdimenA}\else
\if,\@NCCPSextractedtoken\@NCCPStmpdimenA=0.16667\@NCCPStmpdimenA\edef#2{\the\@NCCPStmpdimenA}\else
\if:\@NCCPSextractedtoken\@NCCPStmpdimenA=0.22222\@NCCPStmpdimenA\edef#2{\the\@NCCPStmpdimenA}\else
\if;\@NCCPSextractedtoken\@NCCPStmpdimenA=0.27778\@NCCPStmpdimenA\edef#2{\the\@NCCPStmpdimenA}\else
\if'\@NCCPSextractedtoken\@NCCPStmpdimenA=1\@NCCPStmpdimenA\edef#2{\the\@NCCPStmpdimenA}\else
\if"\@NCCPSextractedtoken\@NCCPStmpdimenA=2\@NCCPStmpdimenA\edef#2{\the\@NCCPStmpdimenA}\else
\if.\@NCCPSextractedtoken\@NCCPStmpdimenA=0.5\arraycolsep\edef#2{\the\@NCCPStmpdimenA}\else
\if/\@NCCPSextractedtoken\edef#2{\the\arraycolsep}\else
\if?\@NCCPSextractedtoken\@NCCPStmpdimenA=2\arraycolsep\edef#2{\the\@NCCPStmpdimenA}\else
\if *\@NCCPSextractedtoken\edef#2{0pt plus 1fil minus 0pt}\else
\if+\@NCCPSextractedtoken\edef#2{\@NCCPSeqnarraycolSEPcenter}\else
\if-\@NCCPSextractedtoken\edef#2{\@NCCPSeqnarraycolSEPzero}\else
\edef#2{\@NCCPSeqnarraycolSEPzero}%
\@NCCPSclspkgerror{Invalid predefined inter-column glue type "#1" in\MessageBreak
column specifications. Using a default value of\MessageBreak
0pt instead}%
{Only !,:;'"./?*+ and - are valid predefined glue types in the\MessageBreak 
NCCPSeqnarray column specifications.}\fi\fi\fi\fi\fi\fi\fi\fi\fi\fi\fi\fi}


% usage: \@NCCPSprocessNcol{digit}
% process a numerical digit from the column specification
% and look up the corresponding user defined glue value
% can transform current type from n to g or a as the user defined glue is acquired
\def\@NCCPSprocessNcol#1{\if\@NCCPSBPprevtype g%
\@NCCPSclspkgerror{Back-to-back inter-column glue specifiers in column\MessageBreak
specifications. Ignoring consecutive glue specifiers\MessageBreak
after the first}%
{You cannot have two or more glue types next to each other\MessageBreak 
in the NCCPSeqnarray column specifications.}%
\let\@NCCPSBPcurtype=a% abort this glue, future digits will be discarded
\@NCCPSBPcurnum=0\relax%
\else% if we previously aborted a glue
\if\@NCCPSBPprevtype a\@NCCPSBPcurnum=0\let\@NCCPSBPcurtype=a%maintain digit abortion
\else%acquire this number
% save the previous type before the numerical digits started
\if\@NCCPSBPprevtype n\else\let\@NCCPSBPprevsavedtype=\@NCCPSBPprevtype\fi%
\multiply\@NCCPSBPcurnum by 10\relax%
\advance\@NCCPSBPcurnum by #1\relax% add in number, \relax is needed to stop TeX's number scan
\if\@NCCPSBPnexttype n\else%close acquisition
\expandafter\ifx\csname @NCCPSeqnarraycolSEPDEF\expandafter\romannumeral\number\@NCCPSBPcurnum\endcsname\@NCCPSeqnarraycolisdefined%
\edef\@NCCPSBPcurglue{\csname @NCCPSeqnarraycolSEP\expandafter\romannumeral\number\@NCCPSBPcurnum\endcsname}%
\else%user glue not defined
\@NCCPSclspkgerror{Invalid user defined inter-column glue type "\number\@NCCPSBPcurnum" in\MessageBreak
column specifications. Using a default value of\MessageBreak
0pt instead}%
{You must define all NCCPSeqnarray numerical inter-column glue types via\MessageBreak
\string\NCCPSeqnarraydefcolsep \space before they are used in column specifications.}%
\edef\@NCCPSBPcurglue{\@NCCPSeqnarraycolSEPzero}%
\fi% glue defined or not
\let\@NCCPSBPcurtype=g% change the type to reflect the acquired glue
\let\@NCCPSBPprevtype=\@NCCPSBPprevsavedtype% restore the prev type before this number glue
\@NCCPSBPcurnum=0\relax%ready for next acquisition
\fi%close acquisition, get glue
\fi%discard or acquire number
\fi%prevtype glue or not
}


% process an acquired glue
% add any acquired column/glue pair to the preamble
\def\@NCCPSprocessGcol{\if\@NCCPSBPprevtype a\let\@NCCPSBPcurtype=a%maintain previous glue abortions
\else
% if this is the start glue, save it, but do nothing else 
% as this is not used in the preamble, but before
\if\@NCCPSBPprevtype s\edef\@NCCPSBPstartglue{\@NCCPSBPcurglue}%
\else%not the start glue
\if\@NCCPSBPprevtype g%ignore if back to back glues
\@NCCPSclspkgerror{Back-to-back inter-column glue specifiers in column\MessageBreak
specifications. Ignoring consecutive glue specifiers\MessageBreak
after the first}%
{You cannot have two or more glue types next to each other\MessageBreak 
in the NCCPSeqnarray column specifications.}%
\let\@NCCPSBPcurtype=a% abort this glue
\else% not a back to back glue
\if\@NCCPSBPprevtype c\relax% if the previoustype was a col, add column/glue pair to preamble
\ifnum\@NCCPSeqnnumcols>0\relax\@NCCPSappendtoksA{&}\fi
\toks0={##}%
% make preamble advance col counter if this environment needs this
\if@advanceNCCPSeqncolcnt\@NCCPSappendtoksA{\global\advance\@NCCPSeqncolcnt by 1\relax}\fi
% insert the column defintion into the preamble, being careful not to expand
% the column definition
\@NCCPSappendtoksA{\tabskip=\@NCCPSBPcurglue}%
\@NCCPSappendNOEXPANDtoksA{\begingroup\csname @NCCPSeqnarraycolPRE}%
\@NCCPSappendtoksA{\@NCCPSBPcurcolname}%
\@NCCPSappendNOEXPANDtoksA{\endcsname}%
\@NCCPSappendtoksA{\the\toks0}%
\@NCCPSappendNOEXPANDtoksA{\relax\relax\relax\relax\relax%
\relax\relax\relax\relax\relax\csname @NCCPSeqnarraycolPOST}%
\@NCCPSappendtoksA{\@NCCPSBPcurcolname}%
\@NCCPSappendNOEXPANDtoksA{\endcsname\relax\relax\relax\relax\relax%
\relax\relax\relax\relax\relax\endgroup}%
\advance\@NCCPSeqnnumcols by 1\relax%one more column in the preamble
\else% error: non-start glue with no pending column
\@NCCPSclspkgerror{Inter-column glue specifier without a prior column\MessageBreak
type in the column specifications. Ignoring this glue\MessageBreak 
specifier}%
{Except for the first and last positions, glue can be placed only\MessageBreak
between column types.}%
\let\@NCCPSBPcurtype=a% abort this glue
\fi% previous was a column
\fi% back-to-back glues
\fi% is start column glue
\fi% prev type not a
}


% process an acquired letter referenced column and, if necessary, add it to the preamble
\def\@NCCPSprocessCcol{\if\@NCCPSBPnexttype g\else
\if\@NCCPSBPnexttype n\else
% we have a column followed by something other than a glue (or numeral glue)
% so we must add this column to the preamble now
\ifnum\@NCCPSeqnnumcols>0\relax\@NCCPSappendtoksA{&}\fi%col separator for those after the first
\if\@NCCPSBPnexttype e\@NCCPSappendtoksA{\tabskip=\@NCCPSBPendglue\relax}\else%put in end glue
\@NCCPSappendtoksA{\tabskip=\@NCCPSeqnarraycolSEPdefaultmid\relax}\fi% or default mid glue
\toks0={##}%
% make preamble advance col counter if this environment needs this
\if@advanceNCCPSeqncolcnt\@NCCPSappendtoksA{\global\advance\@NCCPSeqncolcnt by 1\relax}\fi
% insert the column definition into the preamble, being careful not to expand
% the column definition
\@NCCPSappendNOEXPANDtoksA{\begingroup\csname @NCCPSeqnarraycolPRE}%
\@NCCPSappendtoksA{\@NCCPSBPcurcolname}%
\@NCCPSappendNOEXPANDtoksA{\endcsname}%
\@NCCPSappendtoksA{\the\toks0}%
\@NCCPSappendNOEXPANDtoksA{\relax\relax\relax\relax\relax%
\relax\relax\relax\relax\relax\csname @NCCPSeqnarraycolPOST}%
\@NCCPSappendtoksA{\@NCCPSBPcurcolname}%
\@NCCPSappendNOEXPANDtoksA{\endcsname\relax\relax\relax\relax\relax%
\relax\relax\relax\relax\relax\endgroup}%
\advance\@NCCPSeqnnumcols by 1\relax%one more column in the preamble
\fi%next type not numeral
\fi%next type not glue
}


%%
%% END OF NCCPSeqnarray DEFINITIONS
%%





% set up the running headers and footers
%
% header and footer font and size specifications
\def\@NCCPSheaderstyle{\normalfont\scriptsize}
\def\@NCCPSfooterstyle{\normalfont\scriptsize}
%
% compsoc uses sans-serif headers and footers
\ifCLASSOPTIONcompsoc
  \def\@NCCPSheaderstyle{\normalfont\sffamily\scriptsize}
  \def\@NCCPSfooterstyle{\normalfont\sffamily\scriptsize}
\fi


% standard page style, ps@headings
\def\ps@headings{% default to standard twoside headers, no footers
% will change later if the mode requires otherwise
\def\@oddhead{\hbox{}\@NCCPSheaderstyle\rightmark\hfil\thepage}\relax
\def\@evenhead{\@NCCPSheaderstyle\thepage\hfil\leftmark\hbox{}}\relax
\let\@oddfoot\@empty
\let\@evenfoot\@empty
\ifCLASSOPTIONtechnote
  % technote twoside
  \def\@oddhead{\hbox{}\@NCCPSheaderstyle\leftmark\hfil\thepage}\relax
  \def\@evenhead{\@NCCPSheaderstyle\thepage\hfil\leftmark\hbox{}}\relax
\fi
\ifCLASSOPTIONdraftcls
  % draft footers
  \def\@oddfoot{\@NCCPSfooterstyle\@date\hfil DRAFT}\relax
  \def\@evenfoot{\@NCCPSfooterstyle DRAFT\hfil\@date}\relax
\fi
% oneside
\if@twoside\else
  % standard one side headers
  \def\@oddhead{\hbox{}\@NCCPSheaderstyle\leftmark\hfil\thepage}\relax
  \let\@evenhead\@empty
  \ifCLASSOPTIONdraftcls
    % oneside draft footers
    \def\@oddfoot{\@NCCPSfooterstyle\@date\hfil DRAFT}\relax
    \let\@evenfoot\@empty
  \fi
\fi
% turn off headers for conferences
\ifCLASSOPTIONconference
  \let\@oddhead\@empty
  \let\@evenhead\@empty
\fi
% turn off footers for draftclsnofoot
\ifCLASSOPTIONdraftclsnofoot
  \let\@oddfoot\@empty
  \let\@evenfoot\@empty
\fi}


% title page style, ps@NCCPStitlepagestyle
\def\ps@NCCPStitlepagestyle{% default title page headers, no footers
\def\@oddhead{\hbox{}\@NCCPSheaderstyle\leftmark\hfil\thepage}\relax
\def\@evenhead{\@NCCPSheaderstyle\thepage\hfil\leftmark\hbox{}}\relax
\let\@oddfoot\@empty
\let\@evenfoot\@empty
% will change later if the mode requires otherwise
\ifCLASSOPTIONdraftcls
  % draft footers
  \ifCLASSOPTIONdraftclsnofoot\else
    % but only if not draftclsnofoot
    \def\@oddfoot{\@NCCPSfooterstyle\@date\hfil DRAFT}\relax
    \def\@evenfoot{\@NCCPSfooterstyle DRAFT\hfil\@date}\relax
  \fi
\else
  % all nondraft mode footers
  \if@NCCPSusingpubid
    % for title pages that are using a pubid
    % do not repeat pubid on the title page if using a peer review cover page
    \ifCLASSOPTIONpeerreview\else
      % for noncompsoc papers, the pubid uses footnotesize and
      % is at the same vertical position as where the last baseline would normally be
      \def\@oddfoot{\hbox{}\hss\@NCCPSfooterstyle\footnotesize\raisebox{\footskip}[0pt][0pt]{\@NCCPSpubid}\hss\hbox{}}\relax
      \def\@evenfoot{\hbox{}\hss\@NCCPSfooterstyle\footnotesize\raisebox{\footskip}[0pt][0pt]{\@NCCPSpubid}\hss\hbox{}}\relax
      \ifCLASSOPTIONcompsoc
        % for compsoc papers, the pubid is at the same vertical position as the normal footer
        \def\@oddfoot{\hbox{}\hss\@NCCPSfooterstyle\raisebox{0pt}[0pt][0pt]{\@NCCPSpubid}\hss\hbox{}}\relax
        \def\@evenfoot{\hbox{}\hss\@NCCPSfooterstyle\raisebox{0pt}[0pt][0pt]{\@NCCPSpubid}\hss\hbox{}}\relax
      \fi
    \fi
  \fi
\fi
% turn off headers for conferences
\ifCLASSOPTIONconference
  \let\@oddhead\@empty
  \let\@evenhead\@empty
\fi}


% peer review cover page style, ps@NCCPSpeerreviewcoverpagestyle
\def\ps@NCCPSpeerreviewcoverpagestyle{% default peer review cover no headers, no footers
\let\@oddhead\@empty
\let\@evenhead\@empty
\let\@oddfoot\@empty
\let\@evenfoot\@empty
% will change later if the mode requires otherwise
\ifCLASSOPTIONdraftcls
  % draft footers
  \ifCLASSOPTIONdraftclsnofoot\else
    % but only if not draftclsnofoot
    \def\@oddfoot{\@NCCPSfooterstyle\@date\hfil DRAFT}\relax
    \def\@evenfoot{\@NCCPSfooterstyle DRAFT\hfil\@date}\relax
  \fi
\else
  % all nondraft mode footers
  \if@NCCPSusingpubid
    % for peer review cover pages that are using a pubid
    % for noncompsoc papers, the pubid uses footnotesize and
    % is at the same vertical position as where the last baseline would normally be
    \def\@oddfoot{\hbox{}\hss\@NCCPSfooterstyle\footnotesize\raisebox{\footskip}[0pt][0pt]{\@NCCPSpubid}\hss\hbox{}}\relax
    \def\@evenfoot{\hbox{}\hss\@NCCPSfooterstyle\footnotesize\raisebox{\footskip}[0pt][0pt]{\@NCCPSpubid}\hss\hbox{}}\relax
    \ifCLASSOPTIONcompsoc
      % for compsoc papers, the pubid is at the same vertical position as the normal footer
      \def\@oddfoot{\hbox{}\hss\@NCCPSfooterstyle\raisebox{0pt}[0pt][0pt]{\@NCCPSpubid}\hss\hbox{}}\relax
      \def\@evenfoot{\hbox{}\hss\@NCCPSfooterstyle\raisebox{0pt}[0pt][0pt]{\@NCCPSpubid}\hss\hbox{}}\relax
    \fi
  \fi
\fi}



%% Defines the command for putting the header.
%% Note that all the text is forced into uppercase, if you have some text
%% that needs to be in lower case, for instance et. al., then either manually
%% set \leftmark and \rightmark or use \MakeLowercase{et. al.} within the
%% arguments to \markboth.
%% V1.7b add \protect to work with Babel
\def\markboth#1#2{\def\leftmark{\MakeUppercase{\protect#1}}%
\def\rightmark{\MakeUppercase{\protect#2}}}

\def\today{\ifcase\month\or
    January\or February\or March\or April\or May\or June\or
    July\or August\or September\or October\or November\or December\fi
    \space\number\day, \number\year}




%% CITATION AND BIBLIOGRAPHY COMMANDS
%% 
% 
% 
% Modify Latex2e \@citex to separate citations with "], ["
\def\@citex[#1]#2{%
  \let\@citea\@empty
  \@cite{\@for\@citeb:=#2\do
    {\@citea\def\@citea{], [}%
     \edef\@citeb{\expandafter\@firstofone\@citeb\@empty}%
     \if@filesw\immediate\write\@auxout{\string\citation{\@citeb}}\fi
     \@ifundefined{b@\@citeb}{\mbox{\reset@font\bfseries ?}%
       \G@refundefinedtrue
       \@latex@warning
         {Citation `\@citeb' on page \thepage \space undefined}}%
       {\hbox{\csname b@\@citeb\endcsname}}}}{#1}}

% V1.6 we create hooks for the optional use of Donald Arseneau's
% cite.sty package. cite.sty is "smart" and will notice that the
% following format controls are already defined and will not
% redefine them. The result will be the proper sorting of the
% citation numbers and auto detection of 3 or more entry "ranges" -
% all in NCCPS style:  [1], [2], [5]--[7], [12]
% This also allows for an optional note, i.e., \cite[mynote]{..}.
% If the \cite with note has more than one reference, the note will
% be applied to the last of the listed references. It is generally
% desired that if a note is given, only one reference is listed in
% that \cite.
% Thanks to Mr. Arseneau for providing the required format arguments
% to produce the NCCPS style.
\def\citepunct{], [}
\def\citedash{]--[}

% V1.7 default to using same font for urls made by url.sty
\AtBeginDocument{\csname url@samestyle\endcsname}

% V1.6 class files should always provide these
\def\newblock{\hskip .11em\@plus.33em\@minus.07em}
\let\@openbib@code\@empty
% we do not use \@mkboth, nor alter the page style
\newenvironment{theindex}
               {\if@twocolumn
                  \@restonecolfalse
                \else
                  \@restonecoltrue
                \fi
                \twocolumn[\section*{\indexname}]%
                \parindent\z@
                \parskip\z@ \@plus .3\p@\relax
                \columnseprule \z@
                \columnsep 35\p@
                \let\item\@idxitem}
               {\if@restonecol\onecolumn\else\clearpage\fi}
\newcommand\@idxitem{\par\hangindent 40\p@}
\newcommand\subitem{\@idxitem \hspace*{20\p@}}
\newcommand\subsubitem{\@idxitem \hspace*{30\p@}}
\newcommand\indexspace{\par \vskip 10\p@ \@plus5\p@ \@minus3\p@\relax}



% Provide support for the control entries of NCCPS.bst V1.00 and later.
% V1.7 optional argument allows for a different aux file to be specified in
% order to handle multiple bibliographies. For example, with multibib.sty:
% \newcites{sec}{Secondary Literature}
% \bstctlcite[@auxoutsec]{BSTcontrolhak}
\def\bstctlcite{\@ifnextchar[{\@bstctlcite}{\@bstctlcite[@auxout]}}
\def\@bstctlcite[#1]#2{\@bsphack
  \@for\@citeb:=#2\do{%
    \edef\@citeb{\expandafter\@firstofone\@citeb}%
    \if@filesw\immediate\write\csname #1\endcsname{\string\citation{\@citeb}}\fi}%
  \@esphack}

% \NCCPSnoauxwrite{} allows for citations that do not add to or affect
% the order of the existing citation list. Can be useful for \cite
% within \thanks{}.
\DeclareRobustCommand{\NCCPSnoauxwrite}[1]{\relax
\if@filesw
\@fileswfalse
#1\relax\relax\relax\relax\relax
\@fileswtrue
\else
#1\relax\relax\relax\relax\relax
\fi}

% V1.6 provide a way for a user to execute a command just before 
% a given reference number - used to insert a \newpage to balance
% the columns on the last page
\edef\@NCCPStriggerrefnum{0}   % the default of zero means that
                              % the command is not executed
\def\@NCCPStriggercmd{\newpage}

% allow the user to alter the triggered command
\long\def\NCCPStriggercmd#1{\long\def\@NCCPStriggercmd{#1}}

% allow user a way to specify the reference number just before the
% command is executed
\def\NCCPStriggeratref#1{\@NCCPStmpcountA=#1%
\edef\@NCCPStriggerrefnum{\the\@NCCPStmpcountA}}%

% trigger command at the given reference
\def\@NCCPSbibitemprefix{\@NCCPStmpcountA=\@NCCPStriggerrefnum\relax%
\advance\@NCCPStmpcountA by -1\relax%
\ifnum\c@enumiv=\@NCCPStmpcountA\relax\@NCCPStriggercmd\relax\fi}


\def\@biblabel#1{[#1]}

% compsoc journals and conferences left align the reference numbers
\@NCCPScompsoconly{\def\@biblabel#1{[#1]\hfill}}

% controls bib item spacing
\def\NCCPSbibitemsep{0pt plus .5pt}

\@NCCPScompsocconfonly{\def\NCCPSbibitemsep{0.5\baselineskip plus 0.25\baselineskip minus 0.25\baselineskip}}


\def\thebibliography#1{\section*{\refname}%
    \addcontentsline{toc}{section}{\refname}%
    % V1.6 add some rubber space here and provide a command trigger
    \footnotesize\vskip 0.3\baselineskip plus 0.1\baselineskip minus 0.1\baselineskip%
    \list{\@biblabel{\@arabic\c@enumiv}}%
    {\settowidth\labelwidth{\@biblabel{#1}}%
    \leftmargin\labelwidth
    \advance\leftmargin\labelsep\relax
    \itemsep \NCCPSbibitemsep\relax
    \usecounter{enumiv}%
    \let\p@enumiv\@empty
    \renewcommand\theenumiv{\@arabic\c@enumiv}}%
    \let\@NCCPSlatexbibitem\bibitem%
    \def\bibitem{\@NCCPSbibitemprefix\@NCCPSlatexbibitem}%
\def\newblock{\hskip .11em plus .33em minus .07em}%
% originally:
%   \sloppy\clubpenalty4000\widowpenalty4000%
% by adding the \interlinepenalty here, we make it more
% difficult, but not impossible, for LaTeX to break within a reference.
% The NCCPS almost never breaks a reference (but they do it more often with
% technotes). You may get an underfull vbox warning around the bibliography, 
% but the final result will be much more like what the NCCPS will publish. 
% MDS 11/2000
\ifCLASSOPTIONtechnote\sloppy\clubpenalty4000\widowpenalty4000\interlinepenalty100%
\else\sloppy\clubpenalty4000\widowpenalty4000\interlinepenalty500\fi%
    \sfcode`\.=1000\relax}
\let\endthebibliography=\endlist




% TITLE PAGE COMMANDS
% 
% 
% \NCCPSmembership is used to produce the sublargesize italic font used to indicate author 
% NCCPS membership. compsoc uses a large size sans slant font
\def\NCCPSmembership#1{{\@NCCPSnotcompsoconly{\sublargesize}\normalfont\@NCCPScompsoconly{\sffamily}\textit{#1}}}
 

% \NCCPSauthorrefmark{} produces a footnote type symbol to indicate author affiliation.
% When given an argument of 1 to 9, \NCCPSauthorrefmark{} follows the standard LaTeX footnote
% symbol sequence convention. However, for arguments 10 and above, \NCCPSauthorrefmark{} 
% reverts to using lower case roman numerals, so it cannot overflow. Do note that you 
% cannot use \footnotemark[] in place of \NCCPSauthorrefmark{} within \author as the footnote
% symbols will have been turned off to prevent \thanks from creating footnote marks.
% \NCCPSauthorrefmark{} produces a symbol that appears to LaTeX as having zero vertical
% height - this allows for a more compact line packing, but the user must ensure that
% the interline spacing is large enough to prevent \NCCPSauthorrefmark{} from colliding
% with the text above.
% V1.7 make this a robust command
% V1.8 transmag uses an arabic author affiliation symbol
\ifCLASSOPTIONtransmag
\DeclareRobustCommand*{\NCCPSauthorrefmark}[1]{\raisebox{0pt}[0pt][0pt]{\textsuperscript{\footnotesize #1}}}
\else
\DeclareRobustCommand*{\NCCPSauthorrefmark}[1]{\raisebox{0pt}[0pt][0pt]{\textsuperscript{\footnotesize\ensuremath{\ifcase#1\or *\or \dagger\or \ddagger\or%
    \mathsection\or \mathparagraph\or \|\or **\or \dagger\dagger%
    \or \ddagger\ddagger \else\textsuperscript{\expandafter\romannumeral#1}\fi}}}}
\fi


% FONT CONTROLS AND SPACINGS FOR CONFERENCE MODE AUTHOR NAME AND AFFILIATION BLOCKS
% 
% The default font styles for the author name and affiliation blocks (confmode)
\def\@NCCPSauthorblockNstyle{\normalfont\@NCCPScompsocnotconfonly{\sffamily}\sublargesize}
\def\@NCCPSauthorblockAstyle{\normalfont\@NCCPScompsocnotconfonly{\sffamily}\@NCCPScompsocconfonly{\itshape}\normalsize}
% The default if the user does not use an author block
\def\@NCCPSauthordefaulttextstyle{\normalfont\@NCCPScompsocnotconfonly{\sffamily}\sublargesize}

% adjustment spacing from title (or special paper notice) to author name blocks (confmode)
% can be negative
\def\@NCCPSauthorblockconfadjspace{-0.25em}
% compsoc conferences need more space here
\@NCCPScompsocconfonly{\def\@NCCPSauthorblockconfadjspace{0.75\@NCCPSnormalsizeunitybaselineskip}}

% spacing between name and affiliation blocks (confmode)
% This can be negative.
% The NCCPS doesn't want any added spacing here, but I will leave these
% controls in place in case they ever change their mind.
% Personally, I like 0.75ex.
%\def\@NCCPSauthorblockNtopspace{0.75ex}
%\def\@NCCPSauthorblockAtopspace{0.75ex}
\def\@NCCPSauthorblockNtopspace{0.0ex}
\def\@NCCPSauthorblockAtopspace{0.0ex}
\ifCLASSOPTIONtransmag
% transmag uses one line of space above first affiliation block
\def\@NCCPSauthorblockAtopspace{1\@NCCPSnormalsizeunitybaselineskip}
\fi

% baseline spacing within name and affiliation blocks (confmode)
% must be positive, spacings below certain values will make 
% the position of line of text sensitive to the contents of the
% line above it i.e., whether or not the prior line has descenders, 
% subscripts, etc. For this reason it is a good idea to keep
% these above 2.6ex
\def\@NCCPSauthorblockNinterlinespace{2.6ex}
\def\@NCCPSauthorblockAinterlinespace{2.75ex}

% This tracks the required strut size.
% See the \@NCCPSauthorhalign command for the actual default value used.
\def\@NCCPSauthorblockXinterlinespace{2.7ex}

% variables to retain font size and style across groups
% values given here have no effect as they will be overwritten later
\gdef\@NCCPSSAVESTATEfontsize{10}
\gdef\@NCCPSSAVESTATEfontbaselineskip{12}
\gdef\@NCCPSSAVESTATEfontencoding{OT1}
\gdef\@NCCPSSAVESTATEfontfamily{ptm}
\gdef\@NCCPSSAVESTATEfontseries{m}
\gdef\@NCCPSSAVESTATEfontshape{n}

% saves the current font attributes
\def\@NCCPScurfontSAVE{\global\let\@NCCPSSAVESTATEfontsize\f@size%
\global\let\@NCCPSSAVESTATEfontbaselineskip\f@baselineskip%
\global\let\@NCCPSSAVESTATEfontencoding\f@encoding%
\global\let\@NCCPSSAVESTATEfontfamily\f@family%
\global\let\@NCCPSSAVESTATEfontseries\f@series%
\global\let\@NCCPSSAVESTATEfontshape\f@shape}

% restores the saved font attributes
\def\@NCCPScurfontRESTORE{\fontsize{\@NCCPSSAVESTATEfontsize}{\@NCCPSSAVESTATEfontbaselineskip}%
\fontencoding{\@NCCPSSAVESTATEfontencoding}%
\fontfamily{\@NCCPSSAVESTATEfontfamily}%
\fontseries{\@NCCPSSAVESTATEfontseries}%
\fontshape{\@NCCPSSAVESTATEfontshape}%
\selectfont}


% variable to indicate if the current block is the first block in the column
\newif\if@NCCPSprevauthorblockincol   \@NCCPSprevauthorblockincolfalse


% the command places a strut with height and depth = \@NCCPSauthorblockXinterlinespace
% we use this technique to have complete manual control over the spacing of the lines
% within the halign environment.
% We set the below baseline portion at 30%, the above
% baseline portion at 70% of the total length.
% Responds to changes in the document's \baselinestretch
\def\@NCCPSauthorstrutrule{\@NCCPStmpdimenA\@NCCPSauthorblockXinterlinespace%
\@NCCPStmpdimenA=\baselinestretch\@NCCPStmpdimenA%
\rule[-0.3\@NCCPStmpdimenA]{0pt}{\@NCCPStmpdimenA}}


% blocks to hold the authors' names and affilations. 
% Makes formatting easy for conferences
%
% use real definitions in conference mode
% name block
\def\NCCPSauthorblockN#1{\relax\@NCCPSauthorblockNstyle% set the default text style
\gdef\@NCCPSauthorblockXinterlinespace{0pt}% disable strut for spacer row
% the \expandafter hides the \cr in conditional tex, see the array.sty docs
% for details, probably not needed here as the \cr is in a macro
% do a spacer row if needed
\if@NCCPSprevauthorblockincol\expandafter\@NCCPSauthorblockNtopspaceline\fi
\global\@NCCPSprevauthorblockincoltrue% we now have a block in this column
%restore the correct strut value
\gdef\@NCCPSauthorblockXinterlinespace{\@NCCPSauthorblockNinterlinespace}%
% input the author names
#1%
% end the row if the user did not already
\crcr}
% spacer row for names
\def\@NCCPSauthorblockNtopspaceline{\cr\noalign{\vskip\@NCCPSauthorblockNtopspace}}
%
% affiliation block
\def\NCCPSauthorblockA#1{\relax\@NCCPSauthorblockAstyle% set the default text style
\gdef\@NCCPSauthorblockXinterlinespace{0pt}%disable strut for spacer row
% the \expandafter hides the \cr in conditional tex, see the array.sty docs
% for details, probably not needed here as the \cr is in a macro
% do a spacer row if needed
\if@NCCPSprevauthorblockincol\expandafter\@NCCPSauthorblockAtopspaceline\fi
\global\@NCCPSprevauthorblockincoltrue% we now have a block in this column
%restore the correct strut value
\gdef\@NCCPSauthorblockXinterlinespace{\@NCCPSauthorblockAinterlinespace}%
% input the author affiliations
#1%
% end the row if the user did not already
\crcr
% V1.8 transmag does not use any additional affiliation spacing after the first author
\ifCLASSOPTIONtransmag\gdef\@NCCPSauthorblockAtopspace{0pt}\fi}

% spacer row for affiliations
\def\@NCCPSauthorblockAtopspaceline{\cr\noalign{\vskip\@NCCPSauthorblockAtopspace}}


% allow papers to compile even if author blocks are used in modes other
% than conference or peerreviewca. For such cases, we provide dummy blocks.
\ifCLASSOPTIONconference
\else
   \ifCLASSOPTIONpeerreviewca\else
      % not conference, peerreviewca or transmag mode
      \ifCLASSOPTIONtransmag\else
         \def\NCCPSauthorblockN#1{#1}%
         \def\NCCPSauthorblockA#1{#1}%
      \fi
   \fi
\fi



% we provide our own halign so as not to have to depend on tabular
\def\@NCCPSauthorhalign{\@NCCPSauthordefaulttextstyle% default text style
   \lineskip=0pt\relax% disable line spacing
   \lineskiplimit=0pt\relax%
   \baselineskip=0pt\relax%
   \@NCCPScurfontSAVE% save the current font
   \mathsurround\z@\relax% no extra spacing around math
   \let\\\@NCCPSauthorhaligncr% replace newline with halign friendly one
   \tabskip=0pt\relax% no column spacing
   \everycr{}% ensure no problems here
   \@NCCPSprevauthorblockincolfalse% no author blocks yet
   \def\@NCCPSauthorblockXinterlinespace{2.7ex}% default interline space
   \vtop\bgroup%vtop box
   \halign\bgroup&\relax\hfil\@NCCPScurfontRESTORE\relax ##\relax
   \hfil\@NCCPScurfontSAVE\@NCCPSauthorstrutrule\cr}

% ensure last line, exit from halign, close vbox
\def\end@NCCPSauthorhalign{\crcr\egroup\egroup}

% handle bogus star form
\def\@NCCPSauthorhaligncr{{\ifnum0=`}\fi\@ifstar{\@@NCCPSauthorhaligncr}{\@@NCCPSauthorhaligncr}}

% test and setup the optional argument to \\[]
\def\@@NCCPSauthorhaligncr{\@testopt\@@@NCCPSauthorhaligncr\z@skip}

% end the line and do the optional spacer
\def\@@@NCCPSauthorhaligncr[#1]{\ifnum0=`{\fi}\cr\noalign{\vskip#1\relax}}



% flag to prevent multiple \and warning messages
\newif\if@NCCPSWARNand
\@NCCPSWARNandtrue

% if in conference or peerreviewca modes, we support the use of \and as \author is a
% tabular environment, otherwise we warn the user that \and is invalid
% outside of conference or peerreviewca modes.
\def\and{\relax} % provide a bogus \and that we will then override

\renewcommand{\and}[1][\relax]{\if@NCCPSWARNand\typeout{** WARNING: \noexpand\and is valid only
                               when in conference or peerreviewca}\typeout{modes (line \the\inputlineno).}\fi\global\@NCCPSWARNandfalse}

\ifCLASSOPTIONconference%
\renewcommand{\and}[1][\hfill]{\end{@NCCPSauthorhalign}#1\begin{@NCCPSauthorhalign}}%
\fi
\ifCLASSOPTIONpeerreviewca
\renewcommand{\and}[1][\hfill]{\end{@NCCPSauthorhalign}#1\begin{@NCCPSauthorhalign}}%
\fi
% V1.8 transmag uses conference author format
\ifCLASSOPTIONtransmag
\renewcommand{\and}[1][\hfill]{\end{@NCCPSauthorhalign}#1\begin{@NCCPSauthorhalign}}%
\fi

% page clearing command
% based on LaTeX2e's \cleardoublepage, but allows different page styles
% for the inserted blank pages
\def\@NCCPScleardoublepage#1{\clearpage\if@twoside\ifodd\c@page\else
\hbox{}\thispagestyle{#1}\newpage\if@twocolumn\hbox{}\thispagestyle{#1}\newpage\fi\fi\fi}

% V1.8b hooks to allow adjustment of space above title
\def\NCCPStitletopspace{0.5\baselineskip}
% an added extra amount to allow for adjustment/offset
\def\NCCPStitletopspaceextra{0pt}

% user command to invoke the title page
\def\maketitle{\par%
  \begingroup%
  \normalfont%
  \def\thefootnote{}%  the \thanks{} mark type is empty
  \def\footnotemark{}% and kill space from \thanks within author
  \let\@makefnmark\relax% V1.7, must *really* kill footnotemark to remove all \textsuperscript spacing as well.
  \footnotesize%       equal spacing between thanks lines
  \footnotesep 0.7\baselineskip%see global setting of \footnotesep for more info
  % V1.7 disable \thanks note indention for compsoc
  \@NCCPScompsoconly{\long\def\@makefntext##1{\parindent 1em\noindent\hbox{\@makefnmark}##1}}%
  \normalsize%
  \ifCLASSOPTIONpeerreview
     \newpage\global\@topnum\z@ \@maketitle\@NCCPSstatictitlevskip\@NCCPSaftertitletext%
     \thispagestyle{NCCPSpeerreviewcoverpagestyle}\@thanks%
  \else
     \if@twocolumn%
        \ifCLASSOPTIONtechnote%
           \newpage\global\@topnum\z@ \@maketitle\@NCCPSstatictitlevskip\@NCCPSaftertitletext%
        \else
           \twocolumn[{\NCCPSquantizevspace{\@maketitle}[\NCCPSquantizedisabletitlecmds]{0pt}[-\topskip]{\baselineskip}{\@NCCPSNORMtitlevspace}{\@NCCPSMINtitlevspace}\@NCCPSaftertitletext}]%
        \fi
     \else
        \newpage\global\@topnum\z@ \@maketitle\@NCCPSstatictitlevskip\@NCCPSaftertitletext%
     \fi
     \thispagestyle{NCCPStitlepagestyle}\@thanks%
  \fi
  % pullup page for pubid if used.
  \if@NCCPSusingpubid
     \enlargethispage{-\@NCCPSpubidpullup}%
  \fi 
  \endgroup
  \setcounter{footnote}{0}\let\maketitle\relax\let\@maketitle\relax
  \gdef\@thanks{}%
  % v1.6b do not clear these as we will need the title again for peer review papers
  % \gdef\@author{}\gdef\@title{}%
  \let\thanks\relax}


% V1.8 parbox to format \@NCCPStitleabstractindextext
\long\def\@NCCPStitleabstractindextextbox#1{\parbox{1\textwidth}{#1}}
% V1.8 compsoc is partial width
\ifCLASSOPTIONcompsoc
% comparison with proofs suggests it's in the range of 92.1-92.3%
\long\def\@NCCPStitleabstractindextextbox#1{\parbox{0.922\textwidth}{\@NCCPScompsocnotconfonly{\rightskip\@flushglue\leftskip\z@skip}#1}}
\fi

% formats the Title, authors names, affiliations and special paper notice
% THIS IS A CONTROLLED SPACING COMMAND! Do not allow blank lines or unintentional
% spaces to enter the definition - use % at the end of each line
\def\@maketitle{\newpage
\bgroup\par\vskip\NCCPStitletopspace\vskip\NCCPStitletopspaceextra\centering%
\ifCLASSOPTIONtechnote% technotes, V1.8a abstract and index terms are not treated differently for compsoc technotes
   {\bfseries\large\@NCCPScompsoconly{\Large\sffamily}\@title\par}\vskip 1.3em{\lineskip .5em\@NCCPScompsoconly{\large\sffamily}\@author
   \@NCCPSspecialpapernotice\par}\relax
\else% not a technote
   \vskip0.2em{\Huge\ifCLASSOPTIONtransmag\bfseries\LARGE\fi\@NCCPScompsoconly{\sffamily}\@NCCPScompsocconfonly{\normalfont\normalsize\vskip 2\@NCCPSnormalsizeunitybaselineskip
   \bfseries\Large}\@NCCPScompsocnotconfonly{\vskip 0.75\@NCCPSnormalsizeunitybaselineskip}\@title\par}\relax
   \@NCCPScompsocnotconfonly{\vskip 0.5\@NCCPSnormalsizeunitybaselineskip}\vskip1.0em\par%
   % V1.6 handle \author differently if in conference mode
   \ifCLASSOPTIONconference%
      {\@NCCPSspecialpapernotice\mbox{}\vskip\@NCCPSauthorblockconfadjspace%
       \mbox{}\hfill\begin{@NCCPSauthorhalign}\@author\end{@NCCPSauthorhalign}\hfill\mbox{}\par}\relax
   \else% peerreviewca, peerreview or journal
      \ifCLASSOPTIONpeerreviewca
         % peerreviewca handles author names just like conference mode
         {\@NCCPScompsoconly{\sffamily}\@NCCPSspecialpapernotice\mbox{}\vskip\@NCCPSauthorblockconfadjspace%
          \mbox{}\hfill\begin{@NCCPSauthorhalign}\@author\end{@NCCPSauthorhalign}\hfill\mbox{}\par
          {\@NCCPScompsoconly{\vskip 1.5em\relax
           \@NCCPStitleabstractindextextbox{\@NCCPStitleabstractindextext}\par\noindent\hfill
           \NCCPScompsocdiamondline\hfill\hbox{}\par}}}\relax
      \else% journal, peerreview or transmag
         \ifCLASSOPTIONtransmag
            % transmag also handles author names just like conference mode
            % it also uses \@NCCPStitleabstractindextex, but with one line less
            % space above, and one more below
           {\@NCCPSspecialpapernotice\mbox{}\vskip\@NCCPSauthorblockconfadjspace%
            \mbox{}\hfill\begin{@NCCPSauthorhalign}\@author\end{@NCCPSauthorhalign}\hfill\mbox{}\par
           {\vspace{0.5\baselineskip}\relax\@NCCPStitleabstractindextextbox{\@NCCPStitleabstractindextext}\vspace{-1\baselineskip}\par}}\relax
         \else% journal or peerreview
           {\lineskip.5em\@NCCPScompsoconly{\sffamily}\sublargesize\@author\@NCCPSspecialpapernotice\par
           {\@NCCPScompsoconly{\vskip 1.5em\relax
            \@NCCPStitleabstractindextextbox{\@NCCPStitleabstractindextext}\par\noindent\hfill
            \NCCPScompsocdiamondline\hfill\hbox{}\par}}}\relax
         \fi
      \fi
   \fi
\fi\par\addvspace{0.5\baselineskip}\egroup}


% V1.7 Computer Society "diamond line" which follows index terms for nonconference papers
% V1.8a full width diamond line for single column use
\def\@NCCPScompsocdiamondlinei{\vrule depth 0pt height 0.5pt width 4cm\nobreak\hspace{7.5pt}\nobreak
\raisebox{-3.5pt}{\fontfamily{pzd}\fontencoding{U}\fontseries{m}\fontshape{n}\fontsize{11}{12}\selectfont\char70}\nobreak
\hspace{7.5pt}\nobreak\vrule depth 0pt height 0.5pt width 4cm\relax}
% V1.8a narrower width diamond line for double column use
\def\@NCCPScompsocdiamondlineii{\vrule depth 0pt height 0.5pt width 2.5cm\nobreak\hspace{7.5pt}\nobreak
\raisebox{-3.5pt}{\fontfamily{pzd}\fontencoding{U}\fontseries{m}\fontshape{n}\fontsize{11}{12}\selectfont\char70}\nobreak
\hspace{7.5pt}\nobreak\vrule depth 0pt height 0.5pt width 2.5cm\relax}
% V1.8a bare core without rules to base a last resort on for very narrow linewidths
\def\@NCCPScompsocdiamondlineiii{\mbox{}\nobreak\hspace{7.5pt}\nobreak
\raisebox{-3.5pt}{\fontfamily{pzd}\fontencoding{U}\fontseries{m}\fontshape{n}\fontsize{11}{12}\selectfont\char70}\nobreak
\hspace{7.5pt}\nobreak\mbox{}\relax}

% V1.8a allow \NCCPScompsocdiamondline to adjust for different linewidths.
% Use \@NCCPScompsocdiamondlinei if its width is less than 0.66\linewidth (0.487 nominal for single column)
% if not, fall back to \@NCCPScompsocdiamondlineii if its width is less than 0.75\linewidth (0.659 nominal for double column)
% if all else fails, try to make a custom diamondline based on the abnormally narrow linewidth
\def\NCCPScompsocdiamondline{\settowidth{\@NCCPStmpdimenA}{\@NCCPScompsocdiamondlinei}\relax
\ifdim\@NCCPStmpdimenA<0.66\linewidth\relax\@NCCPScompsocdiamondlinei\relax
\else
\settowidth{\@NCCPStmpdimenA}{\@NCCPScompsocdiamondlineii}\relax
\ifdim\@NCCPStmpdimenA<0.75\linewidth\relax\@NCCPScompsocdiamondlineii\relax
\else
\settowidth{\@NCCPStmpdimenA}{\@NCCPScompsocdiamondlineiii}\relax
\@NCCPStmpdimenB=\linewidth\relax
\addtolength{\@NCCPStmpdimenB}{-1\@NCCPStmpdimenA}\relax
\vrule depth 0pt height 0.5pt width 0.33\@NCCPStmpdimenB\@NCCPScompsocdiamondlineiii\vrule depth 0pt height 0.5pt width 0.33\@NCCPStmpdimenB\relax
\fi\fi}


% V1.7 standard LateX2e \thanks, but with \itshape under compsoc. Also make it a \long\def
% We also need to trigger the one-shot footnote rule
\def\@NCCPStriggeroneshotfootnoterule{\global\@NCCPSenableoneshotfootnoteruletrue}


\long\def\thanks#1{\footnotemark
    \protected@xdef\@thanks{\@thanks
        \protect\footnotetext[\the\c@footnote]{\@NCCPScompsoconly{\itshape
        \protect\@NCCPStriggeroneshotfootnoterule\relax}\ignorespaces#1}}}
\let\@thanks\@empty


% V1.7 allow \author to contain \par's. This is needed to allow \thanks to contain \par.
\long\def\author#1{\gdef\@author{#1}}


% in addition to setting up NCCPSitemize, we need to remove a baselineskip space above and
% below it because \list's \pars introduce blank lines because of the footnote struts.
\def\@NCCPSsetupcompsocitemizelist{\def\labelitemi{$\bullet$}%
\setlength{\NCCPSlabelindent}{0pt}\setlength{\labelsep}{1.2em}\setlength{\parskip}{0pt}%
\setlength{\partopsep}{0pt}\setlength{\topsep}{0.5\baselineskip}\vspace{-1\baselineskip}\relax}


% flag for fake non-compsoc \NCCPScompsocthanksitem - prevents line break on very first item
\newif\if@NCCPSbreakcompsocthanksitem \@NCCPSbreakcompsocthanksitemfalse

\ifCLASSOPTIONcompsoc
% V1.7 compsoc bullet item \thanks
% also, we need to redefine this to destroy the argument in \NCCPSquantizevspace
\long\def\NCCPScompsocitemizethanks#1{\relax\@NCCPSbreakcompsocthanksitemfalse\footnotemark
    \protected@xdef\@thanks{\@thanks
        \protect\footnotetext[\the\c@footnote]{\itshape\protect\@NCCPStriggeroneshotfootnoterule
        {\let\NCCPSiedlistdecl\relax\protect\begin{NCCPSitemize}[\protect\@NCCPSsetupcompsocitemizelist]\ignorespaces#1\relax
        \protect\end{NCCPSitemize}}\protect\vspace{-1\baselineskip}}}}
\DeclareRobustCommand*{\NCCPScompsocthanksitem}{\item}
\else
% non-compsoc, allow for dual compilation via rerouting to normal \thanks
\long\def\NCCPScompsocitemizethanks#1{\thanks{#1}}
% redirect to "pseudo-par" \hfil\break\indent after swallowing [] from \NCCPScompsocthanksitem[]
\DeclareRobustCommand{\NCCPScompsocthanksitem}{\@ifnextchar [{\@NCCPSthanksswallowoptionalarg}%
{\@NCCPSthanksswallowoptionalarg[\relax]}}
% be sure and break only after first item, be sure and ignore spaces after optional argument
\def\@NCCPSthanksswallowoptionalarg[#1]{\relax\if@NCCPSbreakcompsocthanksitem\hfil\break
\indent\fi\@NCCPSbreakcompsocthanksitemtrue\ignorespaces}
\fi


% V1.6b define the \NCCPSpeerreviewmaketitle as needed
\ifCLASSOPTIONpeerreview
\def\NCCPSpeerreviewmaketitle{\@NCCPScleardoublepage{empty}%
\ifCLASSOPTIONtwocolumn
\twocolumn[{\NCCPSquantizevspace{\@NCCPSpeerreviewmaketitle}[\NCCPSquantizedisabletitlecmds]{0pt}[-\topskip]{\baselineskip}{\@NCCPSNORMtitlevspace}{\@NCCPSMINtitlevspace}}]
\else
\newpage\@NCCPSpeerreviewmaketitle\@NCCPSstatictitlevskip
\fi
\thispagestyle{NCCPStitlepagestyle}}
\else
% \NCCPSpeerreviewmaketitle does nothing if peer review option has not been selected
\def\NCCPSpeerreviewmaketitle{\relax}
\fi

% peerreview formats the repeated title like the title in journal papers.
\def\@NCCPSpeerreviewmaketitle{\bgroup\par\addvspace{0.5\baselineskip}\centering\@NCCPScompsoconly{\sffamily}%
\normalfont\normalsize\vskip0.2em{\Huge\@title\par}\vskip1.0em\par
\par\addvspace{0.5\baselineskip}\egroup}



% V1.6 
% this is a static rubber spacer between the title/authors and the main text
% used for single column text, or when the title appears in the first column
% of two column text (technotes). 
\def\@NCCPSstatictitlevskip{{\normalfont\normalsize
% adjust spacing to next text
% v1.6b handle peer review papers
\ifCLASSOPTIONpeerreview
% for peer review papers, the same value is used for both title pages
% regardless of the other paper modes
   \vskip 1\baselineskip plus 0.375\baselineskip minus 0.1875\baselineskip
\else
   \ifCLASSOPTIONconference% conference
      \vskip 1\baselineskip plus 0.375\baselineskip minus 0.1875\baselineskip%
   \else%
      \ifCLASSOPTIONtechnote% technote
         \vskip 1\baselineskip plus 0.375\baselineskip minus 0.1875\baselineskip%
      \else% journal uses more space
         \vskip 2.5\baselineskip plus 0.75\baselineskip minus 0.375\baselineskip%
      \fi
   \fi
\fi}}


% set the nominal and minimum values for the quantized title spacer
% the quantization algorithm will not allow the spacer size to
% become less than \@NCCPSMINtitlevspace - instead it will be lengthened
% default to journal values
\def\@NCCPSNORMtitlevspace{2.5\baselineskip}
\def\@NCCPSMINtitlevspace{2\baselineskip}
% conferences and technotes need tighter spacing
\ifCLASSOPTIONconference% conference
  \def\@NCCPSNORMtitlevspace{1\baselineskip}
  \def\@NCCPSMINtitlevspace{0.75\baselineskip}
\fi
\ifCLASSOPTIONtechnote% technote
  \def\@NCCPSNORMtitlevspace{1\baselineskip}
  \def\@NCCPSMINtitlevspace{0.75\baselineskip}
\fi


% V1.8a
\def\NCCPSquantizevspace{\begingroup\@ifstar{\@NCCPSquantizevspacestarformtrue\@NCCPSquantizevspace}{\@NCCPSquantizevspacestarformfalse\@NCCPSquantizevspace}}
% \NCCPSquantizevspace[output dimen register]{object}[object decl]
%                    {top baselineskip}
%                    [offset][prevdepth][lineskip limit][lineskip]
%                    {unit height}{nominal vspace}{minimum vspace}
%
% Calculates and creates the vspace needed to make the combined height with
% the given object an integer multiple of the given unit height. 
%
% The star form has no effect at present, but is reserved for future use.
%
% If the optional argument [output dimen register] is given, the calculated
% vspace height is stored in the given output dimen (or skip) register
% and no other action is taken, otherwise the object followed by a vspace*
% of the appropriate height is evaluated/output.
%
% The optional object decl (declarations) is code that is evaluated just
% before the object's height is evaluated. Its intented purpose is to allow
% for the alteration or disabling of code within the object during internal
% height evaluation (e.g., \long\def\thanks#1{\relax} ).
% This special code is not invoked if/when the object is rendered at the end.
%
% The nominal vspace is the target value of the added vspace and the minimum
% vspace is the lower allowed limit. The vspacer will be the value that achieves
% integral overall height, in terms of the given unit height, that is closest
% to the nominal vspace and that is not less than the specified minimum vspace.
%
% The line spacing algorithm of TeX is somewhat involved and requires special
% care with regard to the first line of a vertical list (which is indicated
% when \prevdepth is -1000pt or less). top baselineskip specifies the
% baselineskip or topskip used prior to the object. If the height of the
% first line of the object is greater than the given top baselineskip, then
% the top baselineskip is subtracted from the height of the first line and
% that difference is considered along with the rest of the object height
% (because the object will be shifted down by an amount = 
% top line height - top baselineskip). Otherwise, the height of the first line
% of the object is ignored as far as the calculations are concerned.
% This algorithm is adequate for objects that appear at the top of a page
% (e.g., titles) where \topskip spacing is used.
%
% However, as explained on page 78 of the TeXbook, interline spacing is more
% complex when \baselineskip is being used (indicated by \prevdepth > 
% -1000pt). The four optional parameters offset, prevdepth, lineskip limit and
% lineskip are assumed to be equal to be 0pt, \prevdepth, \lineskiplimit and
% \lineskip, respectively, if they are omitted.
%
% The prevdepth is the depth of the line before the object, the lineskip limit
% specifies how close the top of the object can come to the bottom of the
% previous line before \baselineskip is ignored and \lineskip is inserted
% between the object and the line above it. Lineskip does not come into
% play unless the first line of the object is high enough to "get too close"
% (as specified by lineskiplimit) to the line before it. The the prevdepth,
% lineskip limit, and lineskip optional parameters are not needed for the
% first object/line on a page (i.e., prevdepth <= -1000pt) where the simplier
% \topskip spacing rules are in effect.
%
% Offset is a manual adjustment that is added to the height calculations of
% object irrespective of the value of \prevdepth. It is useful when the top
% baselineskip will result in a noninteger unit height object placement even
% if the object itself has integral height. e.g., a footnotesize baselineskip
% is used before the object, thus an offset of, say -3pt, can be given as a
% correction.

% Common combinations of these parameters include:
%
% top baselineskip:   (and default values for offset, prevdepth, etc.)
% \topskip            % for objects that appear at the top of a page
% \maxdimen           % always ignore the height of the top line
% 0pt                 % always consider any positive height of the top line
%
% for objects to appear inline in normal text:
% top baselineskip = \baselineskip
%
% set prevdepth = -1000pt and top baselineskip = 0pt to consider the
% overall height of the object without any other external skip
% consideration

\newif\if@NCCPSquantizevspacestarform       % flag to indicate star form
\newif\if@NCCPSquantizevspaceuseoutdimenreg % flag to indicate output dimen register is to be used
% Use our own private registers because the object could contain a
% structure that uses the existing tmp scratch pad registers
\newdimen\@NCCPSquantizeheightA
\newdimen\@NCCPSquantizeheightB
\newdimen\@NCCPSquantizeheightC
\newdimen\@NCCPSquantizeprevdepth % need to save this early as can change
\newcount\@NCCPSquantizemultiple
\newbox\@NCCPSquantizeboxA


\def\@NCCPSquantizevspace{\@ifnextchar [{\@NCCPSquantizevspaceuseoutdimenregtrue\@@NCCPSquantizevspace}{\@NCCPSquantizevspaceuseoutdimenregfalse\@@NCCPSquantizevspace[]}}


\long\def\@@NCCPSquantizevspace[#1]#2{\relax
% acquire and store
% #1 optional output dimen register
% #2 object
\edef\@NCCPSquantizeoutdimenreg{#1}\relax
% allow for object specifications that contain parameters
\@NCCPStmptoksA={#2}\relax
\long\edef\@NCCPSquantizeobject{\the\@NCCPStmptoksA}\relax
\@ifnextchar [{\@@@NCCPSquantizevspace}{\@@@NCCPSquantizevspace[\relax]}}

\long\def\@@@NCCPSquantizevspace[#1]#2{\relax
% acquire and store
% [#1] optional object decl, is \relax if not given by user
% #2 top baselineskip
% allow for object decl specifications that have parameters
\@NCCPStmptoksA={#1}\relax
\long\edef\@NCCPSquantizeobjectdecl{\the\@NCCPStmptoksA}\relax
\edef\@NCCPSquantizetopbaselineskip{#2}\ivNCCPSquantizevspace}

% acquire optional argument set and store
% [offset][prevdepth][lineskip limit][lineskip]
\def\ivNCCPSquantizevspace{\@ifnextchar [{\@vNCCPSquantizevspace}{\@vNCCPSquantizevspace[0pt]}}
\def\@vNCCPSquantizevspace[#1]{\edef\@NCCPSquantizeoffset{#1}\@ifnextchar [{\@viNCCPSquantizevspace}{\@viNCCPSquantizevspace[\prevdepth]}}
\def\@viNCCPSquantizevspace[#1]{\@NCCPSquantizeprevdepth=#1\relax\@ifnextchar [{\@viiNCCPSquantizevspace}{\@viiNCCPSquantizevspace[\lineskiplimit]}}
\def\@viiNCCPSquantizevspace[#1]{\edef\@NCCPSquantizelineskiplimit{#1}\@ifnextchar [{\@viiiNCCPSquantizevspace}{\@viiiNCCPSquantizevspace[\lineskip]}}
\def\@viiiNCCPSquantizevspace[#1]{\edef\@NCCPSquantizelineskip{#1}\@ixNCCPSquantizevspace}

% main routine
\def\@ixNCCPSquantizevspace#1#2#3{\relax
\edef\@NCCPSquantizeunitheight{#1}\relax
\edef\@NCCPSquantizenomvspace{#2}\relax
\edef\@NCCPSquantizeminvspace{#3}\relax
% \@NCCPSquantizeoutdimenreg
% \@NCCPSquantizeobject
% \@NCCPSquantizeobjectdecl
% \@NCCPSquantizetopbaselineskip
% \@NCCPSquantizeoffset
% \@NCCPSquantizeprevdepth
% \@NCCPSquantizelineskiplimit
% \@NCCPSquantizelineskip
% \@NCCPSquantizeunitheight
% \@NCCPSquantizenomvspace
% \@NCCPSquantizeminvspace
% get overall height of object
\setbox\@NCCPSquantizeboxA\vbox{\begingroup\@NCCPSquantizeobjectdecl\@NCCPSquantizeobject\relax\endgroup}\relax
\@NCCPSquantizeheightA\ht\@NCCPSquantizeboxA\relax
% get height of first line of object
\setbox\@NCCPSquantizeboxA\vtop{\begingroup\@NCCPSquantizeobjectdecl\@NCCPSquantizeobject\relax\endgroup}\relax
\@NCCPSquantizeheightB\ht\@NCCPSquantizeboxA\relax
\ifdim\@NCCPSquantizeprevdepth>-1000pt\relax % prevdepth > -1000pf means full baselineskip\lineskip rules in effect
% lineskip spacing rule takes effect if height of top line > baselineskip - prevdepth - lineskiplimit, 
% otherwise the baselineskip rule is in effect and the height of the first line does not matter at all.
\@NCCPSquantizeheightC=\@NCCPSquantizetopbaselineskip\relax
\advance\@NCCPSquantizeheightC-\@NCCPSquantizeprevdepth\relax
\advance\@NCCPSquantizeheightC-\@NCCPSquantizelineskiplimit\relax % this works even though \@NCCPSquantizelineskiplimit is a macro because TeX allows --10pt notation
\ifdim\@NCCPSquantizeheightB>\@NCCPSquantizeheightC\relax
% lineskip spacing rule is in effect i.e., the object is going to be shifted down relative to the
% baselineskip set position by its top line height (already a part of the total height) + prevdepth + lineskip - baselineskip
\advance\@NCCPSquantizeheightA\@NCCPSquantizeprevdepth\relax
\advance\@NCCPSquantizeheightA\@NCCPSquantizelineskip\relax
\advance\@NCCPSquantizeheightA-\@NCCPSquantizetopbaselineskip\relax
\else
% height of first line <= \@NCCPSquantizetopbaselineskip - \@NCCPSquantizeprevdepth - \@NCCPSquantizelineskiplimit
% standard baselineskip rules are in effect, so don't consider height of first line
\advance\@NCCPSquantizeheightA-\@NCCPSquantizeheightB\relax
\fi
%
\else % prevdepth <= -1000pt, simplier \topskip type rules in effect
\ifdim\@NCCPSquantizeheightB>\@NCCPSquantizetopbaselineskip
% height of top line (already included in the total height) in excess of
% baselineskip is the amount it will be downshifted
\advance\@NCCPSquantizeheightA-\@NCCPSquantizetopbaselineskip\relax
\else
% height of first line is irrelevant, remove it
\advance\@NCCPSquantizeheightA-\@NCCPSquantizeheightB\relax
\fi
\fi % prevdepth <= -1000pt
%
% adjust height for any manual offset
\advance\@NCCPSquantizeheightA\@NCCPSquantizeoffset\relax
% add in nominal spacer
\advance\@NCCPSquantizeheightA\@NCCPSquantizenomvspace\relax
% check for nonzero unitheight
\@NCCPSquantizeheightB=\@NCCPSquantizeunitheight\relax
\ifnum\@NCCPSquantizeheightB=0\relax
\@NCCPSclspkgerror{NCCPSquantizevspace unit height cannot be zero. Assuming 10pt.}%
{Division by zero is not allowed.}
\@NCCPSquantizeheightB=10pt\relax
\fi
% get integer number of lines
\@NCCPSquantizemultiple=\@NCCPSquantizeheightA\relax
\divide\@NCCPSquantizemultiple\@NCCPSquantizeheightB\relax
% set A to contain the excess height over the \@NCCPSquantizemultiple of lines
% A = height - multiple*unitheight
\@NCCPSquantizeheightC\@NCCPSquantizeheightB\relax
\multiply\@NCCPSquantizeheightC\@NCCPSquantizemultiple\relax
\advance\@NCCPSquantizeheightA-\@NCCPSquantizeheightC\relax
% set B to contain the height short of \@NCCPSquantizemultiple+1 of lines
% B = unitheight - A
\advance\@NCCPSquantizeheightB-\@NCCPSquantizeheightA\relax
% choose A or B based on which is closer
\@NCCPSquantizeheightC\@NCCPSquantizenomvspace\relax
\ifdim\@NCCPSquantizeheightA<\@NCCPSquantizeheightB\relax
% C = nomvspace - A, go with lower
\advance\@NCCPSquantizeheightC-\@NCCPSquantizeheightA\relax
\else
% C = nomvspace + B, go with upper
\advance\@NCCPSquantizeheightC\@NCCPSquantizeheightB\relax
\fi
% if violate lower bound, use next integer bound
\ifdim\@NCCPSquantizeheightC<\@NCCPSquantizeminvspace\relax
% A + B = unitheight
\advance\@NCCPSquantizeheightC\@NCCPSquantizeheightA\relax
\advance\@NCCPSquantizeheightC\@NCCPSquantizeheightB\relax
\fi
% export object and spacer outside of group
\global\let\@NCCPSquantizeobjectout\@NCCPSquantizeobject\relax
\global\@NCCPSquantizeheightC\@NCCPSquantizeheightC\relax
\endgroup
\if@NCCPSquantizevspaceuseoutdimenreg
\@NCCPSquantizeoutdimenreg=\@NCCPSquantizeheightC\relax
\else
\@NCCPSquantizeobjectout\relax
\vskip\@NCCPSquantizeheightC\relax
\fi}


% user command to disable all global assignments, possible use within object decl
\def\NCCPSquantizedisableglobal{\let\global\relax
\let\gdef\def
\let\xdef\edef}
% user command to allow for the disabling of \thanks and other commands, possible use within object decl
\def\NCCPSquantizedisabletitlecmds{\long\def\thanks##1{\relax}\relax
\long\def\NCCPScompsocitemizethanks##1{\relax}\def\newpage{\relax}}





% V1.6
% we allow the user access to the last part of the title area
% useful in emergencies such as when a different spacing is needed
% This text is NOT compensated for in the dynamic sizer.
\let\@NCCPSaftertitletext=\relax
\long\def\NCCPSaftertitletext#1{\def\@NCCPSaftertitletext{#1}}


% V1.7 provide a way for users to enter abstract and keywords
% into the onecolumn title are. This text is compensated for
% in the dynamic sizer.
\let\@NCCPStitleabstractindextext=\relax
\long\def\NCCPStitleabstractindextext#1{\def\@NCCPStitleabstractindextext{#1}}

% V1.7 provide a way for users to get the \@NCCPStitleabstractindextext if
% not in compsoc or transmag journal mode - this way abstract and keywords
% can still be placed in their conventional position if not in those modes.
\def\NCCPSdisplaynontitleabstractindextext{%
% display for all conference formats
\ifCLASSOPTIONconference\@NCCPStitleabstractindextext\relax
\else% non-conferences
  % V1.8a display for all technotes
  \ifCLASSOPTIONtechnote\@NCCPStitleabstractindextext\relax
  % V1.8a add diamond line after abstract and index terms for compsoc technotes
  \@NCCPScompsoconly{\noindent\hfill\NCCPScompsocdiamondline\hfill\hbox{}\par}\relax
  \else % non-conferences and non-technotes
    \ifCLASSOPTIONcompsoc% display if not compsoc and not transmag
    \else
      \ifCLASSOPTIONtransmag
      \else% not compsoc journal nor transmag journal
         \@NCCPStitleabstractindextext\relax
      \fi
    \fi
  \fi
\fi}


% command to allow alteration of baselinestretch, but only if the current
% baselineskip is unity. Used to tweak the compsoc abstract and keywords line spacing.
\def\@NCCPStweakunitybaselinestretch#1{{\def\baselinestretch{1}\selectfont
\global\@tempskipa\baselineskip}\ifnum\@tempskipa=\baselineskip%
\def\baselinestretch{#1}\selectfont\fi\relax}


% abstract and keywords are in \small, except 
% for 9pt docs in which they are in \footnotesize
% Because 9pt docs use an 8pt footnotesize, \small
% becomes a rather awkward 8.5pt
\def\@NCCPSabskeysecsize{\small}
\ifx\CLASSOPTIONpt\@NCCPSptsizenine
 \def\@NCCPSabskeysecsize{\footnotesize}
\fi

% compsoc journals use \footnotesize, compsoc conferences use normalsize
\@NCCPScompsoconly{\def\@NCCPSabskeysecsize{\footnotesize}}
\@NCCPScompsocconfonly{\def\@NCCPSabskeysecsize{\small}}


% V1.6 have abstract and keywords strip leading spaces, pars and newlines
% so that spacing is more tightly controlled.
\def\abstract{\normalfont
    \if@twocolumn
      \@NCCPSabskeysecsize\bfseries\textit{\abstractname}---\relax
    \else
      \bgroup\par\addvspace{0.5\baselineskip}\centering\vspace{-1.78ex}\@NCCPSabskeysecsize\textbf{\abstractname}\par\addvspace{0.5\baselineskip}\egroup\quotation\@NCCPSabskeysecsize
    \fi\@NCCPSgobbleleadPARNLSP}
% V1.6 The NCCPS wants only 1 pica from end of abstract to introduction heading when in 
% conference mode (the heading already has this much above it)
\def\endabstract{\relax\ifCLASSOPTIONconference\vspace{0ex}\else\vspace{1.34ex}\fi\par\if@twocolumn\else\endquotation\fi
    \normalfont\normalsize}

\def\NCCPSkeywords{\normalfont
    \if@twocolumn
      \@NCCPSabskeysecsize\bfseries\textit{\NCCPSkeywordsname}---\relax
    \else
      \bgroup\par\addvspace{0.5\baselineskip}\centering\@NCCPSabskeysecsize\textbf{\NCCPSkeywordsname}\par\addvspace{0.5\baselineskip}\egroup\quotation\@NCCPSabskeysecsize
    \fi\@NCCPSgobbleleadPARNLSP}
\def\endNCCPSkeywords{\relax\ifCLASSOPTIONtechnote\vspace{1.34ex}\else\vspace{0.67ex}\fi
    \par\if@twocolumn\else\endquotation\fi%
    \normalfont\normalsize}

% V1.7 compsoc keywords index terms
\ifCLASSOPTIONcompsoc
  \ifCLASSOPTIONconference% compsoc conference
\def\abstract{\normalfont\@NCCPStweakunitybaselinestretch{1.15}\bfseries
    \if@twocolumn
      \@NCCPSabskeysecsize\noindent\textit{\abstractname}---\relax
    \else
      \bgroup\par\addvspace{0.5\baselineskip}\centering\vspace{-1.78ex}\@NCCPSabskeysecsize\textbf{\abstractname}\par\addvspace{0.5\baselineskip}\egroup\quotation\@NCCPSabskeysecsize%
    \fi\@NCCPSgobbleleadPARNLSP}
\def\NCCPSkeywords{\normalfont\@NCCPStweakunitybaselinestretch{1.15}\bfseries
    \if@twocolumn
      \@NCCPSabskeysecsize\vskip 0.5\baselineskip plus 0.25\baselineskip minus 0.25\baselineskip\noindent
      \textit{\NCCPSkeywordsname}---\relax
    \else
      \bgroup\par\addvspace{0.5\baselineskip}\centering\@NCCPSabskeysecsize\textbf{\NCCPSkeywordsname}\par\addvspace{0.5\baselineskip}\egroup\quotation\@NCCPSabskeysecsize%
    \fi\@NCCPSgobbleleadPARNLSP}
  \else% compsoc not conference
\def\abstract{\normalfont\@NCCPStweakunitybaselinestretch{1.15}\sffamily
    \if@twocolumn
      \@NCCPSabskeysecsize\noindent\textbf{\abstractname}---\relax
    \else
      \bgroup\par\addvspace{0.5\baselineskip}\centering\vspace{-1.78ex}\@NCCPSabskeysecsize\textbf{\abstractname}\par\addvspace{0.5\baselineskip}\egroup\quotation\@NCCPSabskeysecsize%
    \fi\@NCCPSgobbleleadPARNLSP}
\def\NCCPSkeywords{\normalfont\@NCCPStweakunitybaselinestretch{1.15}\sffamily
    \if@twocolumn
      \@NCCPSabskeysecsize\vskip 0.5\baselineskip plus 0.25\baselineskip minus 0.25\baselineskip\noindent
      \textbf{\NCCPSkeywordsname}---\relax
    \else
      \bgroup\par\addvspace{0.5\baselineskip}\centering\@NCCPSabskeysecsize\textbf{\NCCPSkeywordsname}\par\addvspace{0.5\baselineskip}\egroup\quotation\@NCCPSabskeysecsize%
    \fi\@NCCPSgobbleleadPARNLSP}
  \fi
\fi

% V1.8 transmag keywords index terms
% no abstract name, use indentation
\ifCLASSOPTIONtransmag
\def\abstract{\normalfont\parindent 1em\relax
    \if@twocolumn
      \@NCCPSabskeysecsize\bfseries\indent
    \else
      \bgroup\par\addvspace{0.5\baselineskip}\centering\vspace{-1.78ex}\@NCCPSabskeysecsize
      \textbf{\abstractname}\par\addvspace{0.5\baselineskip}\egroup\quotation\@NCCPSabskeysecsize
    \fi\@NCCPSgobbleleadPARNLSP}

\def\NCCPSkeywords{\normalfont\parindent 1em\relax
    \if@twocolumn
      \@NCCPSabskeysecsize\vspace{1\baselineskip}\bfseries\indent\textit{\NCCPSkeywordsname}---\relax
    \else
      \bgroup\par\vspace{1\baselineskip}\centering\@NCCPSabskeysecsize
      \textbf{\NCCPSkeywordsname}\par\addvspace{0.5\baselineskip}\egroup\quotation\@NCCPSabskeysecsize
    \fi\@NCCPSgobbleleadPARNLSP}
\fi



% gobbles all leading \, \\ and \par, upon finding first token that
% is not a \ , \\ or a \par, it ceases and returns that token
% 
% used to strip leading \, \\ and \par from the input
% so that such things in the beginning of an environment will not
% affect the formatting of the text
\long\def\@NCCPSgobbleleadPARNLSP#1{\let\@NCCPSswallowthistoken=0%
\let\@NCCPSgobbleleadPARNLSPtoken#1%
\let\@NCCPSgobbleleadPARtoken=\par%
\let\@NCCPSgobbleleadNLtoken=\\%
\let\@NCCPSgobbleleadSPtoken=\ %
\def\@NCCPSgobbleleadSPMACRO{\ }%
\ifx\@NCCPSgobbleleadPARNLSPtoken\@NCCPSgobbleleadPARtoken%
\let\@NCCPSswallowthistoken=1%
\fi%
\ifx\@NCCPSgobbleleadPARNLSPtoken\@NCCPSgobbleleadNLtoken%
\let\@NCCPSswallowthistoken=1%
\fi%
\ifx\@NCCPSgobbleleadPARNLSPtoken\@NCCPSgobbleleadSPtoken%
\let\@NCCPSswallowthistoken=1%
\fi%
% a control space will come in as a macro
% when it is the last one on a line
\ifx\@NCCPSgobbleleadPARNLSPtoken\@NCCPSgobbleleadSPMACRO%
\let\@NCCPSswallowthistoken=1%
\fi%
% if we have to swallow this token, do so and taste the next one
% else spit it out and stop gobbling
\ifx\@NCCPSswallowthistoken 1\let\@NCCPSnextgobbleleadPARNLSP=\@NCCPSgobbleleadPARNLSP\else%
\let\@NCCPSnextgobbleleadPARNLSP=#1\fi%
\@NCCPSnextgobbleleadPARNLSP}%




% TITLING OF SECTIONS
\def\@NCCPSsectpunct{:\ \,}  % Punctuation after run-in section heading  (headings which are
                            % part of the paragraphs), need little bit more than a single space
                            % spacing from section number to title
% compsoc conferences use regular period/space punctuation
\ifCLASSOPTIONcompsoc
\ifCLASSOPTIONconference
\def\@NCCPSsectpunct{.\ }
\fi\fi


\def\@seccntformat#1{\csname the#1dis\endcsname\hskip 0.5em\relax}

\ifCLASSOPTIONcompsoc
% compsoc journals need extra spacing
\ifCLASSOPTIONconference\else
\def\@seccntformat#1{\csname the#1dis\endcsname\hskip 1em\relax}
\fi\fi

%v1.7 put {} after #6 to allow for some types of user font control
%and use \@@par rather than \par
\def\@sect#1#2#3#4#5#6[#7]#8{%
  \ifnum #2>\c@secnumdepth
     \let\@svsec\@empty
  \else
     \refstepcounter{#1}%
     % load section label and spacer into \@svsec
     \protected@edef\@svsec{\@seccntformat{#1}\relax}%
  \fi%
  \@tempskipa #5\relax
  \ifdim \@tempskipa>\z@% tempskipa determines whether is treated as a high
     \begingroup #6{\relax% or low level heading
      \noindent % subsections are NOT indented
       % print top level headings. \@svsec is label, #8 is heading title
       % The NCCPS does not block indent the section title text, it flows like normal
       {\hskip #3\relax\@svsec}{\interlinepenalty \@M #8\@@par}}%
     \endgroup
     \addcontentsline{toc}{#1}{\ifnum #2>\c@secnumdepth\relax\else
               \protect\numberline{\csname the#1\endcsname}\fi#7}%
  \else % printout low level headings
     % svsechd seems to swallow the trailing space, protect it with \mbox{}
     % got rid of sectionmark stuff
     \def\@svsechd{#6{\hskip #3\relax\@svsec #8\@NCCPSsectpunct\mbox{}}%
     \addcontentsline{toc}{#1}{\ifnum #2>\c@secnumdepth\relax\else
               \protect\numberline{\csname the#1\endcsname}\fi#7}}%
  \fi%skip down
  \@xsect{#5}}


% section* handler
%v1.7 put {} after #4 to allow for some types of user font control
%and use \@@par rather than \par
\def\@ssect#1#2#3#4#5{\@tempskipa #3\relax
  \ifdim \@tempskipa>\z@
     %\begingroup #4\@hangfrom{\hskip #1}{\interlinepenalty \@M #5\par}\endgroup
     % The NCCPS does not block indent the section title text, it flows like normal
     \begingroup \noindent #4{\relax{\hskip #1}{\interlinepenalty \@M #5\@@par}}\endgroup
  % svsechd swallows the trailing space, protect it with \mbox{}
  \else \def\@svsechd{#4{\hskip #1\relax #5\@NCCPSsectpunct\mbox{}}}\fi
  \@xsect{#3}}


%% SECTION heading spacing and font
%%
% arguments are: #1 - sectiontype name
% (for \@sect)   #2 - section level
%                #3 - section heading indent
%                #4 - top separation (absolute value used, neg indicates not to indent main text)
%                     If negative, make stretch parts negative too!
%                #5 - (absolute value used) positive: bottom separation after heading,
%                      negative: amount to indent main text after heading
%                Both #4 and #5 negative means to indent main text and use negative top separation
%                #6 - font control
% You've got to have \normalfont\normalsize in the font specs below to prevent
% trouble when you do something like:
% \section{Note}{\ttfamily TT-TEXT} is known to ... 
% The NCCPS sometimes REALLY stretches the area before a section
% heading by up to about 0.5in. However, it may not be a good
% idea to let LaTeX have quite this much rubber.
\ifCLASSOPTIONconference%
% The NCCPS wants section heading spacing to decrease for conference mode
\def\section{\@startsection{section}{1}{\z@}{1.5ex plus 1.5ex minus 0.5ex}%
{0.7ex plus 1ex minus 0ex}{\normalfont\normalsize\centering\scshape}}%
\def\subsection{\@startsection{subsection}{2}{\z@}{1.5ex plus 1.5ex minus 0.5ex}%
{0.7ex plus .5ex minus 0ex}{\normalfont\normalsize\itshape}}%
\else % for journals
\def\section{\@startsection{section}{1}{\z@}{3.0ex plus 1.5ex minus 1.5ex}% V1.6 3.0ex from 3.5ex
{0.7ex plus 1ex minus 0ex}{\normalfont\normalsize\centering\scshape}}%
\def\subsection{\@startsection{subsection}{2}{\z@}{3.5ex plus 1.5ex minus 1.5ex}%
{0.7ex plus .5ex minus 0ex}{\normalfont\normalsize\itshape}}%
\fi

% for both journals and conferences
% decided to put in a little rubber above the section, might help somebody
\def\subsubsection{\@startsection{subsubsection}{3}{\parindent}{0ex plus 0.1ex minus 0.1ex}%
{0ex}{\normalfont\normalsize\itshape}}%
\def\paragraph{\@startsection{paragraph}{4}{2\parindent}{0ex plus 0.1ex minus 0.1ex}%
{0ex}{\normalfont\normalsize\itshape}}%


% compsoc
\ifCLASSOPTIONcompsoc
\ifCLASSOPTIONconference
% compsoc conference
\def\section{\@startsection{section}{1}{\z@}{1\baselineskip plus 0.25\baselineskip minus 0.25\baselineskip}%
{1\baselineskip plus 0.25\baselineskip minus 0.25\baselineskip}{\normalfont\large\bfseries}}%
\def\subsection{\@startsection{subsection}{2}{\z@}{1\baselineskip plus 0.25\baselineskip minus 0.25\baselineskip}%
{1\baselineskip plus 0.25\baselineskip minus 0.25\baselineskip}{\normalfont\sublargesize\bfseries}}%
\def\subsubsection{\@startsection{subsubsection}{3}{\z@}{1\baselineskip plus 0.25\baselineskip minus 0.25\baselineskip}%
{0ex}{\normalfont\normalsize\bfseries}}%
\def\paragraph{\@startsection{paragraph}{4}{2\parindent}{0ex plus 0.1ex minus 0.1ex}%
{0ex}{\normalfont\normalsize}}%
\else% compsoc journals
% use negative top separation as compsoc journals do not indent paragraphs after section titles
\def\section{\@startsection{section}{1}{\z@}{-3.5ex plus -2ex minus -1.5ex}%
{0.7ex plus 1ex minus 0ex}{\normalfont\sublargesize\sffamily\bfseries\scshape}}%
% Note that subsection and smaller may not be correct for the Computer Society,
% I have to look up an example.
\def\subsection{\@startsection{subsection}{2}{\z@}{-3.5ex plus -1.5ex minus -1.5ex}%
{0.7ex plus .5ex minus 0ex}{\normalfont\normalsize\sffamily\bfseries}}%
\def\subsubsection{\@startsection{subsubsection}{3}{\z@}{-2.5ex plus -1ex minus -1ex}%
{0.5ex plus 0.5ex minus 0ex}{\normalfont\normalsize\sffamily\itshape}}%
\def\paragraph{\@startsection{paragraph}{4}{2\parindent}{-0ex plus -0.1ex minus -0.1ex}%
{0ex}{\normalfont\normalsize}}%
\fi\fi

% transmag
\ifCLASSOPTIONtransmag
\def\subsection{\@startsection{subsection}{2}{0.75\parindent}{3.5ex plus 1.5ex minus 1.5ex}%
{0.7ex plus .5ex minus 0ex}{\normalfont\normalsize\itshape}}%
\def\subsubsection{\@startsection{subsubsection}{3}{1.25\parindent}{0.1ex plus 0.1ex minus 0.1ex}%
{0.1ex}{\normalfont\normalsize\itshape}}%
\fi


% V1.8a provide for a raised line Introduction section for use with Computer
% Society papers. We have to remove any spacing glue after the section
% heading and then remove the blank line for the new paragraph after it.
% LaTeX's section handler alters \everypar and we need to propogate those
% changes outside of the \parbox lest there be spacing problems at the top
% of the next section.
\def\NCCPSraisesectionheading#1{\noindent\raisebox{1.5\baselineskip}[0pt][0pt]{\parbox[b]{\columnwidth}{#1\unskip\global\everypar=\everypar}}\vspace{-1\baselineskip}\vspace{-\parskip}\par}



%% ENVIRONMENTS
% "box" symbols at end of proofs
\def\NCCPSQEDclosed{\mbox{\rule[0pt]{1.3ex}{1.3ex}}} % for a filled box
% V1.6 some journals use an open box instead that will just fit around a closed one
\def\NCCPSQEDopen{{\setlength{\fboxsep}{0pt}\setlength{\fboxrule}{0.2pt}\fbox{\rule[0pt]{0pt}{1.3ex}\rule[0pt]{1.3ex}{0pt}}}}
\ifCLASSOPTIONcompsoc
\def\NCCPSQED{\NCCPSQEDopen}   % default to open for compsoc
\else
\def\NCCPSQED{\NCCPSQEDclosed} % otherwise default to closed
\fi

%V1.8 flag to indicate if QED symbol is to be shown
\newif\if@NCCPSQEDshow  \@NCCPSQEDshowtrue
\def\NCCPSproofindentspace{2\parindent}% V1.8 allow user to change indentation amount if desired
% v1.7 name change to avoid namespace collision with amsthm. Also add support
% for an optional argument.
\def\NCCPSproof{\@ifnextchar[{\@NCCPSproof}{\@NCCPSproof[\NCCPSproofname]}}
\def\@NCCPSproof[#1]{\@NCCPSQEDshowtrue\par\noindent\hspace{\NCCPSproofindentspace}{\itshape #1: }}
\def\endNCCPSproof{\if@NCCPSQEDshow\hspace*{\fill}\nobreakspace\NCCPSQED\fi\par}
% qedhere for equation environments, similar to AMS \qedhere
\def\NCCPSQEDhereeqn{\global\@NCCPSQEDshowfalse\eqno\let\eqno\relax\let\leqno\relax
                    \let\veqno\relax\hbox{\NCCPSQED}}
% NCCPS style qedhere for NCCPSeqnarray and other environments
\def\NCCPSQEDhere{\global\@NCCPSQEDshowfalse\NCCPSQED}
% command to disable QED at end of NCCPSproof
\def\NCCPSQEDoff{\global\@NCCPSQEDshowfalse}


%\itemindent is set to \z@ by list, so define new temporary variable
\newdimen\@NCCPStmpitemindent

\ifCLASSOPTIONcompsoc
% V1.8a compsoc uses bold theorem titles, a period instead of a colon, vertical spacing, and hanging indentation
% V1.8 allow long theorem names to break across lines.
% Thanks to Miquel Payaro for reporting this.
\def\@begintheorem#1#2{\@NCCPStmpitemindent\itemindent\relax
    \topsep 0.2\@NCCPSnormalsizeunitybaselineskip plus 0.26\@NCCPSnormalsizeunitybaselineskip minus 0.05\@NCCPSnormalsizeunitybaselineskip
    \rmfamily\trivlist\hangindent\parindent%
    \item[]\textit{\bfseries\noindent #1\ #2.} \itemindent\@NCCPStmpitemindent\relax}
\def\@opargbegintheorem#1#2#3{\@NCCPStmpitemindent\itemindent\relax
\topsep 0.2\@NCCPSnormalsizeunitybaselineskip plus 0.26\@NCCPSnormalsizeunitybaselineskip minus 0.05\@NCCPSnormalsizeunitybaselineskip
\rmfamily\trivlist\hangindent\parindent%
% V1.6 The NCCPS is back to using () around theorem names which are also in italics
% Thanks to Christian Peel for reporting this.
    \item[]\textit{\bfseries\noindent #1\ #2\ (#3).} \itemindent\@NCCPStmpitemindent\relax}
% V1.7 remove bogus \unskip that caused equations in theorems to collide with
% lines below.
\def\@endtheorem{\endtrivlist\vskip 0.25\@NCCPSnormalsizeunitybaselineskip plus 0.26\@NCCPSnormalsizeunitybaselineskip minus 0.05\@NCCPSnormalsizeunitybaselineskip}
\else
%
% noncompsoc
% 
% V1.8 allow long theorem names to break across lines.
% Thanks to Miquel Payaro for reporting this.
\def\@begintheorem#1#2{\@NCCPStmpitemindent\itemindent\relax\topsep 0pt\rmfamily\trivlist%
    \item[]\textit{\indent #1\ #2:} \itemindent\@NCCPStmpitemindent\relax}
\def\@opargbegintheorem#1#2#3{\@NCCPStmpitemindent\itemindent\relax\topsep 0pt\rmfamily \trivlist%
% V1.6 The NCCPS is back to using () around theorem names which are also in italics
% Thanks to Christian Peel for reporting this.
    \item[]\textit{\indent #1\ #2\ (#3):} \itemindent\@NCCPStmpitemindent\relax}
% V1.7 remove bogus \unskip that caused equations in theorems to collide with
% lines below.
\def\@endtheorem{\endtrivlist}
\fi



% V1.6
% display command for the section the theorem is in - so that \thesection
% is not used as this will be in Roman numerals when we want arabic.
% LaTeX2e uses \def\@thmcounter#1{\noexpand\arabic{#1}} for the theorem number
% (second part) display and \def\@thmcountersep{.} as a separator.
% V1.7 intercept calls to the section counter and reroute to \@NCCPSthmcounterinsection
% to allow \appendix(ices} to override as needed.
%
% special handler for sections, allows appendix(ices) to override
\gdef\@NCCPSthmcounterinsection#1{\arabic{#1}}
% string macro
\edef\@NCCPSstringsection{section}

% redefine the #1#2[#3] form of newtheorem to use a hook to \@NCCPSthmcounterinsection
% if section in_counter is used
\def\@xnthm#1#2[#3]{%
  \expandafter\@ifdefinable\csname #1\endcsname
    {\@definecounter{#1}\@newctr{#1}[#3]%
     \edef\@NCCPSstringtmp{#3}
     \ifx\@NCCPSstringtmp\@NCCPSstringsection
     \expandafter\xdef\csname the#1\endcsname{%
     \noexpand\@NCCPSthmcounterinsection{#3}\@thmcountersep
          \@thmcounter{#1}}%
     \else
     \expandafter\xdef\csname the#1\endcsname{%
       \expandafter\noexpand\csname the#3\endcsname \@thmcountersep
          \@thmcounter{#1}}%
     \fi
     \global\@namedef{#1}{\@thm{#1}{#2}}%
     \global\@namedef{end#1}{\@endtheorem}}}



%% SET UP THE DEFAULT PAGESTYLE
\pagestyle{headings}
\pagenumbering{arabic}

% normally the page counter starts at 1
\setcounter{page}{1}
% however, for peerreview the cover sheet is page 0 or page -1
% (for duplex printing)
\ifCLASSOPTIONpeerreview
   \if@twoside
      \setcounter{page}{-1}
   \else
      \setcounter{page}{0}
   \fi
\fi

% standard book class behavior - let bottom line float up and down as
% needed when single sided
\ifCLASSOPTIONtwoside\else\raggedbottom\fi
% if two column - turn on twocolumn, allow word spacings to stretch more and
% enforce a rigid position for the last lines
\ifCLASSOPTIONtwocolumn
% the peer review option delays invoking twocolumn
   \ifCLASSOPTIONpeerreview\else
      \twocolumn
   \fi
\sloppy 
\flushbottom
\fi




% \APPENDIX and \APPENDICES definitions

% This is the \@ifmtarg command from the LaTeX ifmtarg package
% by Peter Wilson (CUA) and Donald Arseneau
% \@ifmtarg is used to determine if an argument to a command
% is present or not.
% For instance:
% \@ifmtarg{#1}{\typeout{empty}}{\typeout{has something}}
% \@ifmtarg is used with our redefined \section command if
% \appendices is invoked.
% The command \section will behave slightly differently depending
% on whether the user specifies a title: 
% \section{My appendix title}
% or not:
% \section{}
% This way, we can eliminate the blank lines where the title
% would be, and the unneeded : after Appendix in the table of
% contents 
\begingroup
\catcode`\Q=3
\long\gdef\@ifmtarg#1{\@xifmtarg#1QQ\@secondoftwo\@firstoftwo\@nil}
\long\gdef\@xifmtarg#1#2Q#3#4#5\@nil{#4}
\endgroup
% end of \@ifmtarg defs


% V1.7
% command that allows the one time saving of the original definition
% of section to \@NCCPSappendixsavesection for \appendix or \appendices 
% we don't save \section here as it may be redefined later by other
% packages (hyperref.sty, etc.)
\def\@NCCPSsaveoriginalsectiononce{\let\@NCCPSappendixsavesection\section
\let\@NCCPSsaveoriginalsectiononce\relax}

% neat trick to grab and process the argument from \section{argument}
% we process differently if the user invoked \section{} with no
% argument (title)
% note we reroute the call to the old \section*
\def\@NCCPSprocessthesectionargument#1{%
\@ifmtarg{#1}{%
\@NCCPSappendixsavesection*{\appendixname\nobreakspace\thesectiondis}%
\addcontentsline{toc}{section}{\appendixname\nobreakspace\thesection}}{%
\@NCCPSappendixsavesection*{\appendixname\nobreakspace\thesectiondis\\* #1}%
\addcontentsline{toc}{section}{\appendixname\nobreakspace\thesection: #1}}}

% we use this if the user calls \section{} after
% \appendix-- which has no meaning. So, we ignore the
% command and its argument. Then, warn the user.
\def\@NCCPSdestroythesectionargument#1{\typeout{** WARNING: Ignoring useless
\protect\section\space in Appendix (line \the\inputlineno).}}


% remember \thesection forms will be displayed in \ref calls
% and in the Table of Contents.
% The \sectiondis form is used in the actual heading itself

% appendix command for one single appendix
% normally has no heading. However, if you want a 
% heading, you can do so via the optional argument:
% \appendix[Optional Heading]
\def\appendix{\relax}
\renewcommand{\appendix}[1][]{\@NCCPSsaveoriginalsectiononce\par
    % v1.6 keep hyperref's identifiers unique
    \gdef\theHsection{Appendix.A}%
    % v1.6 adjust hyperref's string name for the section
    \xdef\Hy@chapapp{appendix}%
    \setcounter{section}{0}%
    \setcounter{subsection}{0}%
    \setcounter{subsubsection}{0}%
    \setcounter{paragraph}{0}%
    \gdef\thesection{A}%
    \gdef\thesectiondis{}% 
    \gdef\thesubsection{\Alph{subsection}}%
    \gdef\@NCCPSthmcounterinsection##1{A}
    \refstepcounter{section}% update the \ref counter
    \@ifmtarg{#1}{\@NCCPSappendixsavesection*{\appendixname}%
                  \addcontentsline{toc}{section}{\appendixname}}{%
             \@NCCPSappendixsavesection*{\appendixname\nobreakspace\\* #1}%
             \addcontentsline{toc}{section}{\appendixname: #1}}%
    % redefine \section command for appendix
    % leave \section* as is
    \def\section{\@ifstar{\@NCCPSappendixsavesection*}{%
                    \@NCCPSdestroythesectionargument}}% throw out the argument
                                                     % of the normal form
}



% appendices command for multiple appendices
% user then calls \section with an argument (possibly empty) to
% declare the individual appendices
\def\appendices{\@NCCPSsaveoriginalsectiononce\par
    % v1.6 keep hyperref's identifiers unique
    \gdef\theHsection{Appendix.\Alph{section}}%
    % v1.6 adjust hyperref's string name for the section
    \xdef\Hy@chapapp{appendix}%
    \setcounter{section}{-1}% we want \refstepcounter to use section 0
    \setcounter{subsection}{0}%
    \setcounter{subsubsection}{0}%
    \setcounter{paragraph}{0}%
    \ifCLASSOPTIONromanappendices%
    \gdef\thesection{\Roman{section}}%
    \gdef\thesectiondis{\Roman{section}}%
    \@NCCPScompsocconfonly{\gdef\thesectiondis{\Roman{section}.}}%
    \gdef\@NCCPSthmcounterinsection##1{A\arabic{##1}}
    \else%
    \gdef\thesection{\Alph{section}}%
    \gdef\thesectiondis{\Alph{section}}%
    \@NCCPScompsocconfonly{\gdef\thesectiondis{\Alph{section}.}}%
    \gdef\@NCCPSthmcounterinsection##1{\Alph{##1}}
    \fi%
    \refstepcounter{section}% update the \ref counter
    \setcounter{section}{0}% NEXT \section will be the FIRST appendix
    % redefine \section command for appendices
    % leave \section* as is
    \def\section{\@ifstar{\@NCCPSappendixsavesection*}{% process the *-form
                    \refstepcounter{section}% or is a new section so,
                    \@NCCPSprocessthesectionargument}}% process the argument 
                                                 % of the normal form
}



% V1.7 compoc uses nonbold drop cap and small caps word style
\ifCLASSOPTIONcompsoc
  \def\NCCPSPARstartFONTSTYLE{\mdseries}
  \def\NCCPSPARstartWORDFONTSTYLE{\scshape}
  \def\NCCPSPARstartWORDCAPSTYLE{\relax}
\fi
%
%
% \NCCPSPARstart
% Definition for the big two line drop cap letter at the beginning of the
% first paragraph of journal papers. The first argument is the first letter
% of the first word, the second argument is the remaining letters of the
% first word which will be rendered in upper case.
% In V1.6 this has been completely rewritten to:
% 
% 1. no longer have problems when the user begins an environment
%    within the paragraph that uses \NCCPSPARstart.
% 2. auto-detect and use the current font family
% 3. revise handling of the space at the end of the first word so that
%    interword glue will now work as normal.
% 4. produce correctly aligned edges for the (two) indented lines.
% 
% We generalize things via control macros - playing with these is fun too.
% 
% V1.7 added more control macros to make it easy for NCCPStools.sty users
% to change the font style.
% 
% the number of lines that are indented to clear it
% may need to increase if using decenders
\providecommand{\NCCPSPARstartDROPLINES}{2}
% minimum number of lines left on a page to allow a \@NCCPSPARstart
% Does not take into consideration rubber shrink, so it tends to
% be overly cautious
\providecommand{\NCCPSPARstartMINPAGELINES}{2}
% V1.7 the height of the drop cap is adjusted to match the height of this text
% in the current font (when \NCCPSPARstart is called).
\providecommand{\NCCPSPARstartHEIGHTTEXT}{T}
% the depth the letter is lowered below the baseline
% the height (and size) of the letter is determined by the sum
% of this value and the height of the \NCCPSPARstartHEIGHTTEXT in the current
% font. It is a good idea to set this value in terms of the baselineskip
% so that it can respond to changes therein.
\providecommand{\NCCPSPARstartDROPDEPTH}{1.1\baselineskip}
% V1.7 the font the drop cap will be rendered in,
% can take zero or one argument.
\providecommand{\NCCPSPARstartFONTSTYLE}{\bfseries}
% V1.7 any additional, non-font related commands needed to modify
% the drop cap letter, can take zero or one argument.
\providecommand{\NCCPSPARstartCAPSTYLE}{\MakeUppercase}
% V1.7 the font that will be used to render the rest of the word,
% can take zero or one argument.
\providecommand{\NCCPSPARstartWORDFONTSTYLE}{\relax}
% V1.7 any additional, non-font related commands needed to modify
% the rest of the word, can take zero or one argument.
\providecommand{\NCCPSPARstartWORDCAPSTYLE}{\MakeUppercase}
% This is the horizontal separation distance from the drop letter to the main text.
% Lengths that depend on the font (e.g., ex, em, etc.) will be referenced
% to the font that is active when \NCCPSPARstart is called. 
\providecommand{\NCCPSPARstartSEP}{0.15em}
% V1.7 horizontal offset applied to the left of the drop cap.
\providecommand{\NCCPSPARstartHOFFSET}{0em}
% V1.7 Italic correction command applied at the end of the drop cap.
\providecommand{\NCCPSPARstartITLCORRECT}{\/}

% width of the letter output, set globally. Can be used in \NCCPSPARstartSEP
% or \NCCPSPARstartHOFFSET, but not the height lengths.
\newdimen\NCCPSPARstartletwidth
\NCCPSPARstartletwidth 0pt\relax

% definition of \NCCPSPARstart
% THIS IS A CONTROLLED SPACING AREA, DO NOT ALLOW SPACES WITHIN THESE LINES
% 
% The token \@NCCPSPARstartfont will be globally defined after the first use
% of \NCCPSPARstart and will be a font command which creates the big letter
% The first argument is the first letter of the first word and the second
% argument is the rest of the first word(s).
\def\NCCPSPARstart#1#2{\par{%
% if this page does not have enough space, break it and lets start
% on a new one
\@NCCPSneedspace{\NCCPSPARstartMINPAGELINES\baselineskip}{\relax}%
% V1.7 move this up here in case user uses \textbf for \NCCPSPARstartFONTSTYLE
% which uses command \leavevmode which causes an unwanted \indent to be issued
\noindent
% calculate the desired height of the big letter
% it extends from the top of \NCCPSPARstartHEIGHTTEXT in the current font
% down to \NCCPSPARstartDROPDEPTH below the current baseline
\settoheight{\@NCCPStmpdimenA}{\NCCPSPARstartHEIGHTTEXT}%
\addtolength{\@NCCPStmpdimenA}{\NCCPSPARstartDROPDEPTH}%
% extract the name of the current font in bold
% and place it in \@NCCPSPARstartFONTNAME
\def\@NCCPSPARstartGETFIRSTWORD##1 ##2\relax{##1}%
{\NCCPSPARstartFONTSTYLE{\selectfont\edef\@NCCPSPARstartFONTNAMESPACE{\fontname\font\space}%
\xdef\@NCCPSPARstartFONTNAME{\expandafter\@NCCPSPARstartGETFIRSTWORD\@NCCPSPARstartFONTNAMESPACE\relax}}}%
% define a font based on this name with a point size equal to the desired
% height of the drop letter
\font\@NCCPSPARstartsubfont\@NCCPSPARstartFONTNAME\space at \@NCCPStmpdimenA\relax%
% save this value as a counter (integer) value (sp points)
\@NCCPStmpcountA=\@NCCPStmpdimenA%
% now get the height of the actual letter produced by this font size
\settoheight{\@NCCPStmpdimenB}{\@NCCPSPARstartsubfont\NCCPSPARstartCAPSTYLE{#1}}%
% If something bogus happens like the first argument is empty or the
% current font is strange, do not allow a zero height.
\ifdim\@NCCPStmpdimenB=0pt\relax%
\typeout{** WARNING: NCCPSPARstart drop letter has zero height! (line \the\inputlineno)}%
\typeout{ Forcing the drop letter font size to 10pt.}%
\@NCCPStmpdimenB=10pt%
\fi%
% and store it as a counter
\@NCCPStmpcountB=\@NCCPStmpdimenB%
% Since a font size doesn't exactly correspond to the height of the capital
% letters in that font, the actual height of the letter, \@NCCPStmpcountB,
% will be less than that desired, \@NCCPStmpcountA
% we need to raise the font size, \@NCCPStmpdimenA 
% by \@NCCPStmpcountA / \@NCCPStmpcountB
% But, TeX doesn't have floating point division, so we have to use integer
% division. Hence the use of the counters.
% We need to reduce the denominator so that the loss of the remainder will
% have minimal affect on the accuracy of the result
\divide\@NCCPStmpcountB by 200%
\divide\@NCCPStmpcountA by \@NCCPStmpcountB%
% Then reequalize things when we use TeX's ability to multiply by
% floating point values
\@NCCPStmpdimenB=0.005\@NCCPStmpdimenA%
\multiply\@NCCPStmpdimenB by \@NCCPStmpcountA%
% \@NCCPSPARstartfont is globaly set to the calculated font of the big letter
% We need to carry this out of the local calculation area to to create the
% big letter.
\global\font\@NCCPSPARstartfont\@NCCPSPARstartFONTNAME\space at \@NCCPStmpdimenB%
% Now set \@NCCPStmpdimenA to the width of the big letter
% We need to carry this out of the local calculation area to set the
% hanging indent
\settowidth{\global\@NCCPStmpdimenA}{\@NCCPSPARstartfont
\NCCPSPARstartCAPSTYLE{#1\NCCPSPARstartITLCORRECT}}}%
% end of the isolated calculation environment
\global\NCCPSPARstartletwidth\@NCCPStmpdimenA\relax%
% add in the extra clearance we want
\advance\@NCCPStmpdimenA by \NCCPSPARstartSEP\relax%
% add in the optional offset
\advance\@NCCPStmpdimenA by \NCCPSPARstartHOFFSET\relax%
% V1.7 don't allow negative offsets to produce negative hanging indents
\@NCCPStmpdimenB\@NCCPStmpdimenA
\ifnum\@NCCPStmpdimenB < 0 \@NCCPStmpdimenB 0pt\fi
% \@NCCPStmpdimenA has the width of the big letter plus the
% separation space and \@NCCPSPARstartfont is the font we need to use
% Now, we make the letter and issue the hanging indent command
% The letter is placed in a box of zero width and height so that other
% text won't be displaced by it.
\hangindent\@NCCPStmpdimenB\hangafter=-\NCCPSPARstartDROPLINES%
\makebox[0pt][l]{\hspace{-\@NCCPStmpdimenA}%
\raisebox{-\NCCPSPARstartDROPDEPTH}[0pt][0pt]{\hspace{\NCCPSPARstartHOFFSET}%
\@NCCPSPARstartfont\NCCPSPARstartCAPSTYLE{#1\NCCPSPARstartITLCORRECT}%
\hspace{\NCCPSPARstartSEP}}}%
{\NCCPSPARstartWORDFONTSTYLE{\NCCPSPARstartWORDCAPSTYLE{\selectfont#2}}}}




% determines if the space remaining on a given page is equal to or greater
% than the specified space of argument one
% if not, execute argument two (only if the remaining space is greater than zero)
% and issue a \newpage
% 
% example: \@NCCPSneedspace{2in}{\vfill}
% 
% Does not take into consideration rubber shrinkage, so it tends to
% be overly cautious
% Based on an example posted by Donald Arseneau
% Note this macro uses \@NCCPStmpdimenB internally for calculations,
% so DO NOT PASS \@NCCPStmpdimenB to this routine
% if you need a dimen register, import with \@NCCPStmpdimenA instead
\def\@NCCPSneedspace#1#2{\penalty-100\begingroup%shield temp variable
\@NCCPStmpdimenB\pagegoal\advance\@NCCPStmpdimenB-\pagetotal% space left
\ifdim #1>\@NCCPStmpdimenB\relax% not enough space left
\ifdim\@NCCPStmpdimenB>\z@\relax #2\fi%
\newpage%
\fi\endgroup}



% NCCPSbiography ENVIRONMENT
% Allows user to enter biography leaving place for picture (adapts to font size)
% As of V1.5, a new optional argument allows you to have a real graphic!
% V1.5 and later also fixes the "colliding biographies" which could happen when a 
% biography's text was shorter than the space for the photo.
% MDS 7/2001
% V1.6 prevent multiple biographies from making multiple TOC entries
\newif\if@NCCPSbiographyTOCentrynotmade
\global\@NCCPSbiographyTOCentrynotmadetrue

% biography counter so hyperref can jump directly to the biographies
% and not just the previous section
\newcounter{NCCPSbiography}
\setcounter{NCCPSbiography}{0}

% photo area size
\def\@NCCPSBIOphotowidth{1.0in}    % width of the biography photo area
\def\@NCCPSBIOphotodepth{1.25in}   % depth (height) of the biography photo area
% area cleared for photo
\def\@NCCPSBIOhangwidth{1.14in}    % width cleared for the biography photo area
\def\@NCCPSBIOhangdepth{1.25in}    % depth cleared for the biography photo area
                                  % actual depth will be a multiple of 
                                  % \baselineskip, rounded up
\def\@NCCPSBIOskipN{4\baselineskip}% nominal value of the vskip above the biography

\newenvironment{NCCPSbiography}[2][]{\normalfont\@NCCPScompsoconly{\sffamily}\footnotesize%
\unitlength 1in\parskip=0pt\par\parindent 1em\interlinepenalty500%
% we need enough space to support the hanging indent
% the nominal value of the spacer
% and one extra line for good measure
\@NCCPStmpdimenA=\@NCCPSBIOhangdepth%
\advance\@NCCPStmpdimenA by \@NCCPSBIOskipN%
\advance\@NCCPStmpdimenA by 1\baselineskip%
% if this page does not have enough space, break it and lets start
% with a new one
\@NCCPSneedspace{\@NCCPStmpdimenA}{\relax}%
% nominal spacer can strech, not shrink use 1fil so user can out stretch with \vfill
\vskip \@NCCPSBIOskipN plus 1fil minus 0\baselineskip%
% the default box for where the photo goes
\def\@NCCPStempbiographybox{{\setlength{\fboxsep}{0pt}\framebox{%
\begin{minipage}[b][\@NCCPSBIOphotodepth][c]{\@NCCPSBIOphotowidth}\centering PLACE\\ PHOTO\\ HERE \end{minipage}}}}%
%
% detect if the optional argument was supplied, this requires the
% \@ifmtarg command as defined in the appendix section above
% and if so, override the default box with what they want
\@ifmtarg{#1}{\relax}{\def\@NCCPStempbiographybox{\mbox{\begin{minipage}[b][\@NCCPSBIOphotodepth][c]{\@NCCPSBIOphotowidth}%
\centering%
#1%
\end{minipage}}}}% end if optional argument supplied
% Make an entry into the table of contents only if we have not done so before
\if@NCCPSbiographyTOCentrynotmade%
% link labels to the biography counter so hyperref will jump
% to the biography, not the previous section
\setcounter{NCCPSbiography}{-1}%
\refstepcounter{NCCPSbiography}%
\addcontentsline{toc}{section}{Biographies}%
\global\@NCCPSbiographyTOCentrynotmadefalse%
\fi%
% one more biography
\refstepcounter{NCCPSbiography}%
% Make an entry for this name into the table of contents 
\addcontentsline{toc}{subsection}{#2}%
% V1.6 properly handle if a new paragraph should occur while the
% hanging indent is still active. Do this by redefining \par so
% that it will not start a new paragraph. (But it will appear to the
% user as if it did.) Also, strip any leading pars, newlines, or spaces.
\let\@NCCPSBIOORGparCMD=\par% save the original \par command
\edef\par{\hfil\break\indent}% the new \par will not be a "real" \par
\settoheight{\@NCCPStmpdimenA}{\@NCCPStempbiographybox}% get height of biography box
\@NCCPStmpdimenB=\@NCCPSBIOhangdepth%
\@NCCPStmpcountA=\@NCCPStmpdimenB% countA has the hang depth
\divide\@NCCPStmpcountA by \baselineskip%  calculates lines needed to produce the hang depth
\advance\@NCCPStmpcountA by 1% ensure we overestimate
% set the hanging indent
\hangindent\@NCCPSBIOhangwidth%
\hangafter-\@NCCPStmpcountA%
% reference the top of the photo area to the top of a capital T
\settoheight{\@NCCPStmpdimenB}{\mbox{T}}%
% set the photo box, give it zero width and height so as not to disturb anything
\noindent\makebox[0pt][l]{\hspace{-\@NCCPSBIOhangwidth}\raisebox{\@NCCPStmpdimenB}[0pt][0pt]{%
\raisebox{-\@NCCPSBIOphotodepth}[0pt][0pt]{\@NCCPStempbiographybox}}}%
% now place the author name and begin the bio text
\noindent\textbf{#2\ }\@NCCPSgobbleleadPARNLSP}{\relax\let\par=\@NCCPSBIOORGparCMD\par%
% 7/2001 V1.5 detect when the biography text is shorter than the photo area
% and pad the unused area - preventing a collision from the next biography entry
% MDS
\ifnum \prevgraf <\@NCCPStmpcountA\relax% detect when the biography text is shorter than the photo
    \advance\@NCCPStmpcountA by -\prevgraf% calculate how many lines we need to pad
    \advance\@NCCPStmpcountA by -1\relax% we compensate for the fact that we indented an extra line
    \@NCCPStmpdimenA=\baselineskip% calculate the length of the padding
    \multiply\@NCCPStmpdimenA by \@NCCPStmpcountA%
    \noindent\rule{0pt}{\@NCCPStmpdimenA}% insert an invisible support strut
\fi%
\par\normalfont}



% V1.6
% added biography without a photo environment
\newenvironment{NCCPSbiographynophoto}[1]{%
% Make an entry into the table of contents only if we have not done so before
\if@NCCPSbiographyTOCentrynotmade%
% link labels to the biography counter so hyperref will jump
% to the biography, not the previous section
\setcounter{NCCPSbiography}{-1}%
\refstepcounter{NCCPSbiography}%
\addcontentsline{toc}{section}{Biographies}%
\global\@NCCPSbiographyTOCentrynotmadefalse%
\fi%
% one more biography
\refstepcounter{NCCPSbiography}%
% Make an entry for this name into the table of contents 
\addcontentsline{toc}{subsection}{#1}%
\normalfont\@NCCPScompsoconly{\sffamily}\footnotesize\interlinepenalty500%
\vskip 4\baselineskip plus 1fil minus 0\baselineskip%
\parskip=0pt\par%
\noindent\textbf{#1\ }\@NCCPSgobbleleadPARNLSP}{\relax\par\normalfont}


% provide the user with some old font commands
% got this from article.cls
\DeclareOldFontCommand{\rm}{\normalfont\rmfamily}{\mathrm}
\DeclareOldFontCommand{\sf}{\normalfont\sffamily}{\mathsf}
\DeclareOldFontCommand{\tt}{\normalfont\ttfamily}{\mathtt}
\DeclareOldFontCommand{\bf}{\normalfont\bfseries}{\mathbf}
\DeclareOldFontCommand{\it}{\normalfont\itshape}{\mathit}
\DeclareOldFontCommand{\sl}{\normalfont\slshape}{\@nomath\sl}
\DeclareOldFontCommand{\sc}{\normalfont\scshape}{\@nomath\sc}
\DeclareRobustCommand*\cal{\@fontswitch\relax\mathcal}
\DeclareRobustCommand*\mit{\@fontswitch\relax\mathnormal}


% SPECIAL PAPER NOTICE COMMANDS
% 
% holds the special notice text
\def\@NCCPSspecialpapernotice{\relax}
 
% for special papers, like invited papers, the user can do:
% \NCCPSspecialpapernotice{(Invited Paper)} before \maketitle
\def\NCCPSspecialpapernotice#1{\ifCLASSOPTIONconference%
\def\@NCCPSspecialpapernotice{{\sublargesize\textit{#1}\vspace*{1em}}}%
\else%
\def\@NCCPSspecialpapernotice{{\\*[1.5ex]\sublargesize\textit{#1}}\vspace*{-2ex}}%
\fi}




% PUBLISHER ID COMMANDS
% to insert a publisher's ID footer
% V1.6 \NCCPSpubid has been changed so that the change in page size and style
% occurs in \maketitle. \NCCPSpubid must now be issued prior to \maketitle
% use \NCCPSpubidadjcol as before - in the second column of the title page
% These changes allow \maketitle to take the reduced page height into
% consideration when dynamically setting the space between the author 
% names and the maintext.
%
% the amount the main text is pulled up to make room for the
% publisher's ID footer
% The NCCPS uses about 1.3\baselineskip for journals, 
% dynamic title spacing will clean up the fraction
\def\@NCCPSpubidpullup{1.3\baselineskip}
\ifCLASSOPTIONtechnote
% for technotes it must be an integer of baselineskip as there can be no
% dynamic title spacing for two column mode technotes (the title is in the
% in first column) and we should maintain an integer number of lines in the
% second column
% There are some examples (such as older issues of "Transactions on
% Information Theory") in which the NCCPS really pulls the text off the ID for
% technotes - about 0.55in (or 4\baselineskip). We'll use 2\baselineskip
% and call it even.
\def\@NCCPSpubidpullup{2\baselineskip}
\fi

% V1.7 compsoc does not use a pullup
\ifCLASSOPTIONcompsoc
\def\@NCCPSpubidpullup{0pt}
\fi

% holds the ID text
\def\@NCCPSpubid{\relax}

% flag so \maketitle can tell if \NCCPSpubid was called
\newif\if@NCCPSusingpubid
\global\@NCCPSusingpubidfalse
% issue this command in the page to have the ID at the bottom
% V1.6 use before \maketitle
\def\NCCPSpubid#1{\def\@NCCPSpubid{#1}\global\@NCCPSusingpubidtrue}


% command which will pull up (shorten) the column it is executed in
% to make room for the publisher ID. Place in the second column of
% the title page when using \NCCPSpubid
% Is smart enough not to do anything when in single column text or
% if the user hasn't called \NCCPSpubid
% currently needed in for the second column of a page with the
% publisher ID. If not needed in future releases, please provide this
% command and define it as \relax for backward compatibility
% v1.6b do not allow command to operate if the peer review option has been 
% selected because \NCCPSpubidadjcol will not be on the cover page.
% V1.7 do nothing if compsoc
\def\NCCPSpubidadjcol{\ifCLASSOPTIONcompsoc\else\ifCLASSOPTIONpeerreview\else
\if@twocolumn\if@NCCPSusingpubid\enlargethispage{-\@NCCPSpubidpullup}\fi\fi\fi\fi}

% Special thanks to Peter Wilson, Daniel Luecking, and the other
% gurus at comp.text.tex, for helping me to understand how best to
% implement the NCCPSpubid command in LaTeX.



%% Lockout some commands under various conditions

% general purpose bit bucket
\newsavebox{\@NCCPSrubishbin}

% flags to prevent multiple warning messages
\newif\if@NCCPSWARNthanks
\newif\if@NCCPSWARNNCCPSPARstart
\newif\if@NCCPSWARNNCCPSbiography
\newif\if@NCCPSWARNNCCPSbiographynophoto
\newif\if@NCCPSWARNNCCPSpubid
\newif\if@NCCPSWARNNCCPSpubidadjcol
\newif\if@NCCPSWARNNCCPSmembership
\newif\if@NCCPSWARNNCCPSaftertitletext
\@NCCPSWARNthankstrue
\@NCCPSWARNNCCPSPARstarttrue
\@NCCPSWARNNCCPSbiographytrue
\@NCCPSWARNNCCPSbiographynophototrue
\@NCCPSWARNNCCPSpubidtrue
\@NCCPSWARNNCCPSpubidadjcoltrue
\@NCCPSWARNNCCPSmembershiptrue
\@NCCPSWARNNCCPSaftertitletexttrue


%% Lockout some commands when in various modes, but allow them to be restored if needed
%%
% save commands which might be locked out
% so that the user can later restore them if needed
\let\@NCCPSSAVECMDthanks\thanks
\let\@NCCPSSAVECMDNCCPSPARstart\NCCPSPARstart
\let\@NCCPSSAVECMDNCCPSbiography\NCCPSbiography
\let\@NCCPSSAVECMDendNCCPSbiography\endNCCPSbiography
\let\@NCCPSSAVECMDNCCPSbiographynophoto\NCCPSbiographynophoto
\let\@NCCPSSAVECMDendNCCPSbiographynophoto\endNCCPSbiographynophoto
\let\@NCCPSSAVECMDNCCPSpubid\NCCPSpubid
\let\@NCCPSSAVECMDNCCPSpubidadjcol\NCCPSpubidadjcol
\let\@NCCPSSAVECMDNCCPSmembership\NCCPSmembership
\let\@NCCPSSAVECMDNCCPSaftertitletext\NCCPSaftertitletext


% disable \NCCPSPARstart when in draft mode
% This may have originally been done because the pre-V1.6 drop letter
% algorithm had problems with a non-unity baselinestretch
% At any rate, it seems too formal to have a drop letter in a draft
% paper.
\ifCLASSOPTIONdraftcls
\def\NCCPSPARstart#1#2{#1#2\if@NCCPSWARNNCCPSPARstart\typeout{** ATTENTION: \noexpand\NCCPSPARstart
 is disabled in draft mode (line \the\inputlineno).}\fi\global\@NCCPSWARNNCCPSPARstartfalse}
\fi
% and for technotes
\ifCLASSOPTIONtechnote
\def\NCCPSPARstart#1#2{#1#2\if@NCCPSWARNNCCPSPARstart\typeout{** WARNING: \noexpand\NCCPSPARstart
 is locked out for technotes (line \the\inputlineno).}\fi\global\@NCCPSWARNNCCPSPARstartfalse}
\fi


% lockout unneeded commands when in conference mode
\ifCLASSOPTIONconference
% when locked out, \thanks, \NCCPSbiography, \NCCPSbiographynophoto, \NCCPSpubid,
% \NCCPSmembership and \NCCPSaftertitletext will all swallow their given text. 
% \NCCPSPARstart will output a normal character instead
% warn the user about these commands only once to prevent the console screen
% from filling up with redundant messages
\def\thanks#1{\if@NCCPSWARNthanks\typeout{** WARNING: \noexpand\thanks
 is locked out when in conference mode (line \the\inputlineno).}\fi\global\@NCCPSWARNthanksfalse}
\def\NCCPSPARstart#1#2{#1#2\if@NCCPSWARNNCCPSPARstart\typeout{** WARNING: \noexpand\NCCPSPARstart
 is locked out when in conference mode (line \the\inputlineno).}\fi\global\@NCCPSWARNNCCPSPARstartfalse}


% LaTeX treats environments and commands with optional arguments differently.
% the actual ("internal") command is stored as \\commandname 
% (accessed via \csname\string\commandname\endcsname )
% the "external" command \commandname is a macro with code to determine
% whether or not the optional argument is presented and to provide the 
% default if it is absent. So, in order to save and restore such a command
% we would have to save and restore \\commandname as well. But, if LaTeX
% ever changes the way it names the internal names, the trick would break.
% Instead let us just define a new environment so that the internal
% name can be left undisturbed.
\newenvironment{@NCCPSbogusbiography}[2][]{\if@NCCPSWARNNCCPSbiography\typeout{** WARNING: \noexpand\NCCPSbiography
 is locked out when in conference mode (line \the\inputlineno).}\fi\global\@NCCPSWARNNCCPSbiographyfalse%
\setbox\@NCCPSrubishbin\vbox\bgroup}{\egroup\relax}
% and make biography point to our bogus biography
\let\NCCPSbiography=\@NCCPSbogusbiography
\let\endNCCPSbiography=\end@NCCPSbogusbiography

\renewenvironment{NCCPSbiographynophoto}[1]{\if@NCCPSWARNNCCPSbiographynophoto\typeout{** WARNING: \noexpand\NCCPSbiographynophoto
 is locked out when in conference mode (line \the\inputlineno).}\fi\global\@NCCPSWARNNCCPSbiographynophotofalse%
\setbox\@NCCPSrubishbin\vbox\bgroup}{\egroup\relax}

\def\NCCPSpubid#1{\if@NCCPSWARNNCCPSpubid\typeout{** WARNING: \noexpand\NCCPSpubid 
 is locked out when in conference mode (line \the\inputlineno).}\fi\global\@NCCPSWARNNCCPSpubidfalse}
\def\NCCPSpubidadjcol{\if@NCCPSWARNNCCPSpubidadjcol\typeout{** WARNING: \noexpand\NCCPSpubidadjcol
 is locked out when in conference mode (line \the\inputlineno).}\fi\global\@NCCPSWARNNCCPSpubidadjcolfalse}
\def\NCCPSmembership#1{\if@NCCPSWARNNCCPSmembership\typeout{** WARNING: \noexpand\NCCPSmembership
 is locked out when in conference mode (line \the\inputlineno).}\fi\global\@NCCPSWARNNCCPSmembershipfalse}
\def\NCCPSaftertitletext#1{\if@NCCPSWARNNCCPSaftertitletext\typeout{** WARNING: \noexpand\NCCPSaftertitletext
 is locked out when in conference mode (line \the\inputlineno).}\fi\global\@NCCPSWARNNCCPSaftertitletextfalse}
\fi


% provide a way to restore the commands that are locked out
\def\NCCPSoverridecommandlockouts{%
\typeout{** ATTENTION: Overriding command lockouts (line \the\inputlineno).}%
\let\thanks\@NCCPSSAVECMDthanks%
\let\NCCPSPARstart\@NCCPSSAVECMDNCCPSPARstart%
\let\NCCPSbiography\@NCCPSSAVECMDNCCPSbiography%
\let\endNCCPSbiography\@NCCPSSAVECMDendNCCPSbiography%
\let\NCCPSbiographynophoto\@NCCPSSAVECMDNCCPSbiographynophoto%
\let\endNCCPSbiographynophoto\@NCCPSSAVECMDendNCCPSbiographynophoto%
\let\NCCPSpubid\@NCCPSSAVECMDNCCPSpubid%
\let\NCCPSpubidadjcol\@NCCPSSAVECMDNCCPSpubidadjcol%
\let\NCCPSmembership\@NCCPSSAVECMDNCCPSmembership%
\let\NCCPSaftertitletext\@NCCPSSAVECMDNCCPSaftertitletext}



% need a backslash character for typeout output
{\catcode`\|=0 \catcode`\\=12
|xdef|@NCCPSbackslash{\}}


% hook to allow easy disabling of all legacy warnings
\def\@NCCPSlegacywarn#1#2{\typeout{** ATTENTION: \@NCCPSbackslash #1 is deprecated (line \the\inputlineno).
Use \@NCCPSbackslash #2 instead.}}


% provide some legacy NCCPS commands
\def\NCCPScompsoctitleabstractindextext{\@NCCPSlegacywarn{NCCPScompsoctitleabstractindextext}{NCCPStitleabstractindextext}\NCCPStitleabstractindextext}
\def\NCCPSdisplaynotcompsoctitleabstractindextext{\@NCCPSlegacywarn{NCCPSdisplaynotcompsoctitleabstractindextext}{NCCPSdisplaynontitleabstractindextext}\NCCPSdisplaynontitleabstractindextext}
% provide some legacy NCCPS environments


% V1.8a no more support for these legacy commands
%\def\authorblockA{\@NCCPSlegacywarn{authorblockA}{NCCPSauthorblockA}\NCCPSauthorblockA}
%\def\authorblockN{\@NCCPSlegacywarn{authorblockN}{NCCPSauthorblockN}\NCCPSauthorblockN}
%\def\authorrefmark{\@NCCPSlegacywarn{authorrefmark}{NCCPSauthorrefmark}\NCCPSauthorrefmark}
%\def\PARstart{\@NCCPSlegacywarn{PARstart}{NCCPSPARstart}\NCCPSPARstart}
%\def\pubid{\@NCCPSlegacywarn{pubid}{NCCPSpubid}\NCCPSpubid}
%\def\pubidadjcol{\@NCCPSlegacywarn{pubidadjcol}{NCCPSpubidadjcol}\NCCPSpubidadjcol}
%\def\specialpapernotice{\@NCCPSlegacywarn{specialpapernotice}{NCCPSspecialpapernotice}\NCCPSspecialpapernotice}
% and environments
%\def\keywords{\@NCCPSlegacywarn{keywords}{NCCPSkeywords}\NCCPSkeywords}
%\def\endkeywords{\endNCCPSkeywords}
% V1.8 no more support for legacy IED list commands
%\let\labelindent\NCCPSlabelindent
%\def\calcleftmargin{\@NCCPSlegacywarn{calcleftmargin}{NCCPScalcleftmargin}\NCCPScalcleftmargin}
%\def\setlabelwidth{\@NCCPSlegacywarn{setlabelwidth}{NCCPSsetlabelwidth}\NCCPSsetlabelwidth}
%\def\usemathlabelsep{\@NCCPSlegacywarn{usemathlabelsep}{NCCPSusemathlabelsep}\NCCPSusemathlabelsep}
%\def\iedlabeljustifyc{\@NCCPSlegacywarn{iedlabeljustifyc}{NCCPSiedlabeljustifyc}\NCCPSiedlabeljustifyc}
%\def\iedlabeljustifyl{\@NCCPSlegacywarn{iedlabeljustifyl}{NCCPSiedlabeljustifyl}\NCCPSiedlabeljustifyl}
%\def\iedlabeljustifyr{\@NCCPSlegacywarn{iedlabeljustifyr}{NCCPSiedlabeljustifyr}\NCCPSiedlabeljustifyr}
% V1.8 no more support for QED and proof stuff
%\def\QED{\@NCCPSlegacywarn{QED}{NCCPSQED}\NCCPSQED}
%\def\QEDclosed{\@NCCPSlegacywarn{QEDclosed}{NCCPSQEDclosed}\NCCPSQEDclosed}
%\def\QEDopen{\@NCCPSlegacywarn{QEDopen}{NCCPSQEDopen}\NCCPSQEDopen}
%\AtBeginDocument{\def\proof{\@NCCPSlegacywarn{proof}{NCCPSproof}\NCCPSproof}\def\endproof{\endNCCPSproof}}
% V1.8 no longer support biography or biographynophoto
%\def\biography{\@NCCPSlegacywarn{biography}{NCCPSbiography}\NCCPSbiography}
%\def\biographynophoto{\@NCCPSlegacywarn{biographynophoto}{NCCPSbiographynophoto}\NCCPSbiographynophoto}
%\def\endbiography{\endNCCPSbiography}
%\def\endbiographynophoto{\endNCCPSbiographynophoto}
% V1.7 and later no longer supports \overrideNCCPSmargins
%\def\overrideNCCPSs{%
%\typeout{** WARNING: \string\overrideNCCPSmargins \space no longer supported (line \the\inputlineno).}%
%\typeout{** Use the \string\CLASSINPUTinnersidemargin, \string\CLASSINPUToutersidemargin \space controls instead.}}

\endinput

%%%%%%%%%%%%%%%%%%%%%%%%%%%%% End of NCCPS.cls  %%%%%%%%%%%%%%%%%%%%%%%%%%%%
% That's all folks!

